function y = ExpliziterEuler(f, t0, T, h, y0, params, precision)
% function y = ExpliziterEuler(f, t0, T, h, y0, params, precision)
%
% Implementiert das explizite Euler-Verfahren zur Loesung des AWPs
%     y' = f(t,y(t))
%   y(0) = y0
% im Intervall [t0,T].
%
% Input:
%   --
%   f ... Zeiger auf die rechte Seite des AWPs
%   t0 ... Anfangszeitpunkt
%   T ... Endzeitpunkt
%   h ... Schrittweite
%   y0 ... Anfangswert
%   params ... Zusaetzliche Parameter fuer f
%   precision ... Genauigkeit (fuer Demo-Zwecke)
%
% Output:
%   --
%   y ... Berechnete Loesung. In der ersten Zeile 
%         sind die Zeitschritte gespeichert, in den 
%         restlichen Zeilen die zugehoerigen 
%         Funktionswerte der Loesung.
%

% Anzahl der Zeitschritte bestimmen:
nsteps = ceil((T-t0)/h);

% y initialisieren
if nargin > 6 && strcmp(precision, 's') 
    % Rechnung mit single precision
    y = single(zeros(length(y0)+1, nsteps+1));
else    
    % Rechnung mit double precision (Standard)
    y = zeros(length(y0)+1, nsteps+1);
end

% Den Startwert kennen wir schon
y(1,1) = t0;
y(2:end,1) = y0;

% Schleife ueber alle Zeitschritte:
for i=1:nsteps
    % Zeitpunkt aktualisieren:
    % (Durch Rundungsfehler koennte man ueber
    % T hinausschiessen, das wird hier abgefragt.)
    y(1,i+1) = y(1,i) + min(T-y(1,i),h); 
    h = y(1,i+1) - y(1,i);
    
    % Auswertung des expliziten Euler-Verfahrens
    if isempty(params) 
        % keine weiteren Parameter an f uebergeben
        y(2:end,i+1) ...
          = y(2:end,i) + h*feval(f, t0 + (i-1)*h, y(2:end,i));
    else
        % f braucht zusaetzliche Parameter 
        % (z.B. Raeuber-Beute-Modell)
        y(2:end,i+1) ...
          = y(2:end,i) + h*feval(f, t0 + (i-1)*h, y(2:end,i), params);
    end
end
