function [y, nNewton] = implMidpoint(f, t0, T, h, y0, nmax, method,epsi,A)
% function y = implMidpoint(f, t0, T, h, y0, params, nmax)
%
% Implementiert das impl. MP-Verfahren zur Loesung des AWPs
%     y' = f(t,y(t))
%   y(0) = y0
% im Intervall [t0,T].
%
% Input:
%   --
%   f ... Zeiger auf die rechte Seite des AWPs
%   t0 ... Anfangszeitpunkt
%   T ... Endzeitpunkt
%   h ... Schrittweite
%   y0 ... Anfangswert
%   params ... Zusaetzliche Parameter fuer f
%   nmax ... Maximale Anzahl der Newton-Schritte pro Zeitschritt
%   method ... 'Newton' oder 'Vereinfacht'
%
% Output:
%   --
%   y ... Berechnete Loesung. In der ersten Zeile
%         sind die Zeitschritte gespeichert, in den
%         restlichen Zeilen die zugehoerigen
%         Funktionswerte der Loesung.
%   nNewton ... Anzahl der tatsächlich benötigten Newton-Schritte pro
%         Zeitschritt
%

% Anzahl der Zeitschritte bestimmen:
nsteps = ceil((T-t0)/h);

% y initialisieren
y = zeros(length(y0)+1, nsteps+1);

% Den Startwert kennen wir schon
y(1,1) = t0;
y(2:end,1) = y0;

nNewton = zeros(1,size(y,2)-1);
% Schleife ueber alle Zeitschritte:
for i=1:nsteps
    % Zeitpunkt aktualisieren:
    % (Durch Rundungsfehler koennte man ueber
    % T hinausschiessen, das wird hier abgefragt.)
    y(1,i+1) = y(1,i) + min(T-y(1,i),h);
    h = y(1,i+1) - y(1,i);
    
    nNewtoni = 0;
    
    % f braucht zusaetzliche Parameter
    % (z.B. Raeuber-Beute-Modell)
    if strcmpi(method, 'newton')
        k = zeros(size(y(2:end,i)));
        % Auswerten der Funktion und ihrer Ableitung
        [F,DF] = feval(f,y(1,i)+h/2, y(2:end,i)+h/2*k,A,epsi);
        % Das Residuum
        Fz = k- F;
        while (norm(Fz) > 1e-6 || nNewtoni < 1) && nNewtoni < nmax
            % Lösen der Newton-Gleichung
            dk = -(eye(size(k,1)) - h/2*DF)\Fz;
            % Update
            k = k + dk;
            % Alles wieder von vorne
            [F,DF] = feval(f,y(1,i)+h/2, y(2:end,i)+h/2*k,A,epsi);
            Fz = k- F;
            nNewtoni = nNewtoni + 1;
        end
        y(2:end,i+1) = y(2:end,i)+h*k;
    end
    nNewton(i) = nNewtoni;
end
