close all;
clear all;

% This program computes the solution to a two space fraction plus one time
% dimensional differential equation. For reference please cite
%
% Low-rank solvers for fractional differential equations. Tobias Breiten,
% Valeria Simoncini, and Martin Stoll, To appear in ETNA, 2016
%
% This code uses synthetic right-hand sides.
%
% 2016, Stoll, Breiten, Simoncini

%fractional parameters
alpha = .5; beta = 1.7; gammad = 1.7;

L = 1;   % length of spatial interval
nx = 2^10;     % Number of spatial steps of discretization
ny = 2^10;
nt = 2^3;     % Number of steps in time
hx = L/(nx+1);          % spatial step
hy = L/(ny+1);


trunctol =1e-12; % truncation tolerance
tol = 1e-10; % tolerance for the outer solver
tolIT = 1e-10; % tolerance for the iterative solver within kpiiik


tau = hx*hy;     % time step

% Coefficient vectors
v_alpha = cumprod([1, 1 - ((alpha+1)./ (1:nt))]);
v_beta =  cumprod([1, 1 - ((beta+1)./(1:nx))]);
v_gammad = cumprod([1, 1 - ((gammad+1)./(1:ny))]);

lalpha1 = (tau^(-alpha))*v_alpha(1:end)';
lalpha2 = (tau^(-alpha))*[v_alpha(1);zeros(nt-1,1)];


% coefficients
ap = spdiags(gamma(1.2)*((1:nx)'*hx).^beta,0,nx,nx);
am = spdiags(gamma(1.2)*(2-(1:nx)'*hx).^beta,0,nx,nx);
bp = spdiags(gamma(1.2)*((1:ny)'*hy).^gammad,0,ny,ny);
bm = spdiags(gamma(1.2)*(2-(1:ny)'*hy).^gammad,0,ny,ny);


% Space-derivative of symmetric Riesz type computed from a shifted
% Gruenwald approximation
lbeta = hx^(-beta)*(v_beta)'; % symmetric Riesz approximation
lgammad = hy^(-gammad)*v_gammad'; % symmetric Riesz approximation


[X,Y] = meshgrid(hx:hx:L-hx,hy:hy:L-hy);

F = 10*exp(10*X).*sin(Y.*X); % Setting the forcing term time-independent

% Apply the boundary conditions in 1D, first and last entry are
% on the boundary and all remaning degrees of freedom are interior

% Adjusting the forcing term
F=F';
F=F(:);

U(:,1)=zeros((nx)*(ny),1);
U11{1} =[];
U12{1} =[];


% Setting up the preconditioner
ex=ones(nx,1);
ey=ones(ny,1);
if beta>=1.5
    Dx = spdiags([-1*ex 2*ex -1*ex],-1:1,nx,nx);
elseif beta<=1.5
    Dx = spdiags([1*ex -1*ex],0:1,nx,nx);
end
if gammad>=1.5
    Dy = spdiags([-1*ey 2*ey -1*ey],-1:1,ny,ny);
elseif gammad<=1.5
    Dy = spdiags([1*ey -1*ey],0:1,ny,ny);
end

[La,Ua]=lu(1/2*speye(nx)+tau/(hx^beta)*(ap*Dx+am*Dx'));
[Lb,Ub]=lu(1/2*speye(ny)+tau/(hy^gammad)*(bp*Dy+bm*Dy'));

nA = 0;
nP = 0;
lCA=[];
lCB=[];
tic
for i=1:nt
    i
    % time-dependent forcing term (better choices welcome)
    F = (100*sin(10*X).*cos(Y))+100*sin(100*i*tau).*X.*Y; % Setting the forcing term time-dependent
    % Adjusting the forcing term
    F=F';
    [B11,B12]=trunc([U11{i}  tau*reshape(F,nx,ny)],[U12{i} speye(ny)],trunctol);
    
    %% IKPIK solver
    [U11{i+1},U12{i+1},res]=ikpik_mf_prob3_coeff(lbeta(2:end),[lbeta(1);zeros(nx-2,1)],lgammad(2:end),[lgammad(1);zeros(ny-2,1)],lCA,lCB,-B11,B12,100,tol,tolIT,La,Ua,Lb,Ub,ap,am,bp,bm,tau);
    vex(i) = size(res,2);
    
end
toc
%% Plotting the solution (Uncomment if you like to see the solution)

% for i=1:nt
%     %             T=U(:,i);
%     T=U11{i+1}*U12{i+1}';
%     axis([0 ny 0 nx -.1 .1])
%     %             surf(reshape(T,nx-2,ny-2),'LineStyle','none'),
%     surf(reshape(T,nx,ny),'LineStyle','none'),
%     %             hold on
%     pause
% end

