% This program computes the solution to a three space plus one time
% dimensional fraction differential equation. For reference please cite
% 
% Low-rank solvers for fractional differential equations. Tobias Breiten, 
% Valeria Simoncini, and Martin Stoll, To appear in ETNA, 2016
%
% This codes reqires the TT toolbox. Usingthe qtt_toep methodology will
% allow to avoid the toeplitz matrices and hence can tackle much higher
% dimensions.
%
% 
% 2016, Stoll, Breiten, Simoncini


close all;
clear all;

% fractional parameters
alpha = .9; beta = 1.7; gamma = 1.1; delta=1.7;
L = 1;   % length of spatial interval
nx = 2^6;     % Number of spatial steps of discretization
ny = 2^7;
nz = 2^6;
nt = 2^6;     % Number of steps in time
hx = L / (nx+1);          % spatial step
hy = L / (ny+1);
hz = L / (nz+1);

trunctol =1e-8; % truncation tolerance
tol = 1e-6; % tolerance for the outer linear solver

tau = hx*hy*hz;     % time step

% Coefficient vectors
v_alpha = cumprod([1, 1 - ((alpha+1) ./ (1:nt+1))]);
v_beta =  cumprod([1, 1 - ((beta+1) ./ (1:nx))]);
v_gamma = cumprod([1, 1 - ((gamma+1) ./ (1:ny))]);
v_delta = cumprod([1, 1 - ((delta+1) ./ (1:nz))]);

% Toeplitz matrices these are dense and can be avoided by using the
% qtt_toep function !!! to be done
Talpha = (tau^(-alpha))*(toeplitz(v_alpha(1:end-1),[v_alpha(1);zeros(nt,1)])); % Time-derivative of Caputo type
Cbeta = (hx^(-beta))*toeplitz(v_beta(2:end),[v_beta(2);v_beta(1);zeros(nx-2,1)]);
Cgamma = (hy^(-gamma))*toeplitz(v_gamma(2:end),[v_gamma(2);v_gamma(1);zeros(ny-2,1)]);
Cdelta = (hz^(-delta))*toeplitz(v_delta(2:end),[v_delta(2);v_delta(1);zeros(nz-2,1)]);
Lbeta  = 1/2*(Cbeta+Cbeta');
Lgamma = 1/2*(Cgamma+Cgamma');
Ldelta = 1/2*(Cdelta+Cdelta');

Int = speye(nt+1);
Inx = speye(nx);
Iny = speye(ny);
Inz = speye(nz);

xmesh = hx:hx:L-hx;
ymesh = hy:hy:L-hy;
zmesh = hz:hz:L-hz;

mx = (nx);   % dimension of G_j matrix
my = (ny); % dimension of K_j matrix
mz = (nz); % dimension of K_j matrix
mt = (nt+1); % dimension of K_j matrix

ttspace =tkron(tkron(tt_matrix(reshape(Lbeta, [mx,mx]),1e-16),tt_eye(my)),tt_eye(mz))...
    +tkron(tkron(tt_eye(mx),tt_matrix( reshape(Lgamma, [my,my]),1e-16)),tt_eye(mz))...
    +tkron(tkron(tt_eye(mx),tt_eye(my)),tt_matrix( reshape(Ldelta, [mz,mz]),1e-16));
Part1 = tkron(tkron(tkron(tt_eye(mx),tt_eye(my)),tt_eye(mz)),tt_matrix(reshape(Talpha,[mt,mt]),1e-16));
Part2 = tkron(ttspace,tt_eye(mt));
ttall=round(Part1-Part2,trunctol);

ttrhs = tkron(tkron(tkron(tt_tensor(exp((xmesh-0.5).^2),1e-16, [mx]),tt_tensor(exp((ymesh-0.5).^2),1e-16,my)),tt_tensor(exp((zmesh-0.5).^2),1e-16,mz)),tt_ones(mt));

sol=amen_solve2(ttall,ttrhs,tol,'trunc_norm', 'fro', 'max_full_size', 1, 'local_restart', 80);

break;
