function [U,S,V,flag] = mysvds(U11,U12,n1,n2)
%SVDS   Find a few singular values and vectors.
%   If A is M-by-N, SVDS(A,...) manipulates a few eigenvalues and vectors
%   returned by EIGS(B,...), where B = [SPARSE(M,M) A; A' SPARSE(N,N)],
%   to find a few singular values and vectors of A.  The positive
%   eigenvalues of the symmetric matrix B are the same as the singular
%   values of A.
%
%   S = SVDS(A) returns the 6 largest singular values of A.
%
%   S = SVDS(A,K) computes the K largest singular values of A.
%
%   S = SVDS(A,K,SIGMA) computes the K singular values closest to the
%   scalar shift SIGMA.  For example, S = SVDS(A,K,0) computes the K
%   smallest singular values.
%
%   S = SVDS(A,K,'L') computes the K largest singular values (the default).
%
%   S = SVDS(A,K,SIGMA,OPTIONS) sets some parameters (see EIGS):
%
%   Field name       Parameter                               Default
%
%   OPTIONS.tol      Convergence tolerance:                    1e-10
%                    NORM(A*V-U*S,1) <= tol * NORM(A,1).
%   OPTIONS.maxit    Maximum number of iterations.               300
%   OPTIONS.disp     Number of values displayed each iteration.    0
%
%   [U,S,V] = SVDS(A,...) computes the singular vectors as well.
%   If A is M-by-N and K singular values are computed, then U is M-by-K
%   with orthonormal columns, S is K-by-K diagonal, and V is N-by-K with
%   orthonormal columns.
%
%   [U,S,V,FLAG] = SVDS(A,...) also returns a convergence flag.
%   If EIGS converged then NORM(A*V-U*S,1) <= TOL * NORM(A,1) and
%   FLAG is 0.  If EIGS did not converge, then FLAG is 1.
%
%   Note: SVDS is best used to find a few singular values of a large,
%   sparse matrix.  To find all the singular values of such a matrix,
%   SVD(FULL(A)) will usually perform better than SVDS(A,MIN(SIZE(A))).
%
%   Example:
%      load west0479
%      sf = svd(full(west0479))
%      sl = svds(west0479,10)
%      ss = svds(west0479,10,0)
%      s2 = svds(west0479,10,2)
%
%      sl will be a vector of the 10 largest singular values, ss will be a
%      vector of the 10 smallest singular values, and s2 will be a vector
%      of the 10 singular values of west0479 which are closest to 2.
%
%   See also SVD, EIGS.

%   Copyright 1984-2011 The MathWorks, Inc.
%   $Revision: 1.16.4.7 $  $Date: 2011/07/05 19:03:40 $

m=n1;
n=n2;
p = min(m,n);
q = max(m,n);

% B's positive eigenvalues are the singular values of A
% "top" of B's eigenvectors correspond to left singular values of A
% "bottom" of B's eigenvectors correspond to right singular vectors of A

k = min(p,40);
% k = p;
bsigma = 'LA';
boptions.tol = 1e-10 / sqrt(2);
boptions.disp = 0;
bk = min(p,k);
[W,D,bflag] = eigs(@(x)truncprod(x,U11,U12,n1,n2),n1+n2,bk);

% permute D and W so diagonal entries of D are sorted by proximity to sigma
d = diag(D);
if strcmp(bsigma,'LA') || strcmp(bsigma,'LR')
    nA = max(d);
    [~,ind] = sort(nA-d);
else
    nA = normest(A);
    [~,ind] = sort(abs(d-bsigma));
end
d = d(ind);
W = W(:,ind);

% Tolerance to determine the "small" singular values of A.
% If eigs did not converge, give extra leeway.
if bflag
    dtol = q * nA * sqrt(eps);
    uvtol = m * sqrt(sqrt(eps));
else
    dtol = q * nA * eps;
    uvtol = m * sqrt(eps);
end
% Which (left singular) vectors are already orthogonal, with norm 1/sqrt(2)?
UU = W(1:m,:)' * W(1:m,:);
dUU = diag(UU);
VV = W(m+(1:n),:)' * W(m+(1:n),:);
dVV = diag(VV);
indpos = find((d > dtol) & (abs(dUU-0.5) <= uvtol) & (abs(dVV-0.5) <= uvtol));
indpos = indpos(1:min(end,k));
npos = length(indpos);
U = sqrt(2) * W(1:m,indpos);
s = d(indpos);
V = sqrt(2) * W(m+(1:n),indpos);

% There may be 2*(p-rank(A)) zero eigenvalues of B corresponding
% to the rank deficiency of A and up to q-p zero eigenvalues
% of B corresponding to the difference between m and n.

if npos < k
    indzero = find(abs(d) <= dtol);
    QWU = orth(W(1:m,indzero));
    QWV = orth(W(m+(1:n),indzero));
    nzero = min([size(QWU,2), size(QWV,2), k-npos]);
    U = [U QWU(:,1:nzero)];
    s = [s; abs(d(indzero(1:nzero)))];
    V = [V QWV(:,1:nzero)];
end

% sort the singular values in descending order (as in svd)
[s,ind] = sort(s);
s = s(end:-1:1);
if nargout <= 1
    U = s;
else
    U = U(:,ind(end:-1:1));
    S = diag(s);
    V = V(:,ind(end:-1:1));
    flag = 33;
%     flag = norm(A*V-U*S,1) > sqrt(2) * boptions.tol * norm(A,1);
end
end

function y=truncprod(v,U11,U12,n1,n2)
% function y=truncprod(v)
w=U12'*v(n1+1:n1+n2,1);
y(1:n1,1)=U11*w;
w=U11'*v(1:n1,1);
y(n1+1:n1+n2,1)=U12*w;
end
