% Numerical experiment on 2-layer stochastic block model (SBM) graphs [2]
% with and without one noisy layer
% for the Allen-Cahn multiclass classification scheme [1, Algorithm 6.1]
% using the power mean Laplacian [3].
% Data generated by the routine generate_sbm_graph [4].
%
% [1] Kai Bergermann, Martin Stoll, and Toni Volkmer. Semi-supervised Learning for Multilayer Graphs Using Diffuse Interface Methods and Fast Matrix Vector Products. Submitted, 2020. 
% [2] P. W. Holland, K. B. Laskey, and S. Leinhardt, Stochastic blockmodels: First steps, Social networks, 5 (1983), pp. 109–137.
% [3] Pedro Mercado, Antoine Gautier, Francesco Tudisco, and Matthias Hein. The power mean Laplacian for multilayer graph clustering. In Proceedings of the Twenty-First International Conference on Artificial Intelligence and Statistics, volume 84 of Proceedings of Machine Learning Research, pages 1828-1838, 2018.
% [4] P. Mercado, F. Tudisco, and M. Hein, Generalized matrix means for semi-supervised learning with multilayer graphs, 2019.
%
% This software is distributed under the GNU General Public License v2. See 
% COPYING for the full license text. If that file is not available, see 
% <http://www.gnu.org/licenses/>.
%
% Copyright (c) 2020 Kai Bergermann

clear all
addpath('../Subroutines')

%% create label information
n_cluster=50;
m=2; T=2;
n=n_cluster*m;
clusters1=[ones(n_cluster,1);zeros(n_cluster,1)];
clusters2=[zeros(n_cluster,1);ones(n_cluster,1)];

%% create true label matrix
Y_mat=[clusters1,clusters2];

%% create initial conditions U0
U0=Y_mat;
perc_damaged=0.96;
for i=1:n_cluster
    if(i<=n_cluster*perc_damaged)
        U0(i,:)=(1/m)*ones(m,1);
        U0(i+n_cluster,:)=(1/m)*ones(m,1);
    end
end

%% SBM, generate weight matrices
p_in=.7; p_out=.3; p_noise=.5;
n_graphs=100;
accuracy_PML_p_pos=zeros(n_graphs,1);
accuracy_PML_p_1=zeros(n_graphs,1);
accuracy_PML_p_minus_20=zeros(n_graphs,1);
accuracy_PML_p_minus_30=zeros(n_graphs,1);
accuracy_PML_p_pos_noise=zeros(n_graphs,1);
accuracy_PML_p_1_noise=zeros(n_graphs,1);
accuracy_PML_p_minus_20_noise=zeros(n_graphs,1);
accuracy_PML_p_minus_30_noise=zeros(n_graphs,1);
rng('default');
rng(2406);

for j=1:n_graphs
    [W1,~]=generate_sbm_graph(clusters1, p_in, p_out);

    % case 1: Clustering-Information in both layers
    [W2,~]=generate_sbm_graph(clusters2, p_in, p_out);

    % case 2: second layer noisy
    [W2_noise,~]=generate_sbm_graph(clusters2, p_noise, p_noise);

    %% Plot sparsity pattern of the weight matrices
    subplot(131)
    spy(W1)
    subplot(132)
    spy(W2)
    subplot(133)
    spy(W2_noise)

    %% Create single layer graph Laplacians
    D1invsqrt=diag(sum(W1).^-0.5);
    D2invsqrt=diag(sum(W2).^-0.5);
    D2_noise_invsqrt=diag(sum(W2_noise).^-0.5);

    %%
    p1=10;
    delta=log(1+abs(p1));
    if(p1==1)
        L1=eye(n)-D1invsqrt*W1*D1invsqrt;
        L2=eye(n)-D2invsqrt*W2*D2invsqrt;
        L2_noise=eye(n)-D2_noise_invsqrt*W2_noise*D2_noise_invsqrt;
        Lp1=(1/T)*(L1+L2);
        Lp1_noise=(1/T)*(L1+L2_noise);
    elseif(p1>0)
        L1=eye(n)-D1invsqrt*W1*D1invsqrt;
        L2=eye(n)-D2invsqrt*W2*D2invsqrt;
        L2_noise=eye(n)-D2_noise_invsqrt*W2_noise*D2_noise_invsqrt;
        Lp1=((1/T)*(L1^p1+L2^p1))^(1/p1);
        Lp1_noise=((1/T)*(L1^p1+L2_noise^p1))^(1/p1);
    else
        L1=(1+delta)*eye(n)-D1invsqrt*W1*D1invsqrt;
        L2=(1+delta)*eye(n)-D2invsqrt*W2*D2invsqrt;
        L2_noise=(1+delta)*eye(n)-D2_noise_invsqrt*W2_noise*D2_noise_invsqrt;
        Lp1=((1/T)*(L1^p1+L2^p1))^(1/p1);
        Lp1_noise=((1/T)*(L1^p1+L2_noise^p1))^(1/p1);
    end

    k=2;
    [phi1,lambda1]=eigs(Lp1,k,'sm');
    [phi1_noise,lambda1_noise]=eigs(Lp1_noise,k,'sm');

    %%
    p2=1;
    delta=log(1+abs(p2));
    if(p2==1)
        L1=eye(n)-D1invsqrt*W1*D1invsqrt;
        L2=eye(n)-D2invsqrt*W2*D2invsqrt;
        L2_noise=eye(n)-D2_noise_invsqrt*W2_noise*D2_noise_invsqrt;
        Lp2=(1/T)*(L1+L2);
        Lp2_noise=(1/T)*(L1+L2_noise);
    elseif(p2>0)
        L1=eye(n)-D1invsqrt*W1*D1invsqrt;
        L2=eye(n)-D2invsqrt*W2*D2invsqrt;
        L2_noise=eye(n)-D2_noise_invsqrt*W2_noise*D2_noise_invsqrt;
        Lp2=((1/T)*(L1^p2+L2^p2))^(1/p2);
        Lp2_noise=((1/T)*(L1^p2+L2_noise^p2))^(1/p2);
    else
        L1=(1+delta)*eye(n)-D1invsqrt*W1*D1invsqrt;
        L2=(1+delta)*eye(n)-D2invsqrt*W2*D2invsqrt;
        L2_noise=(1+delta)*eye(n)-D2_noise_invsqrt*W2_noise*D2_noise_invsqrt;
        Lp2=((1/T)*(L1^p2+L2^p2))^(1/p2);
        Lp2_noise=((1/T)*(L1^p2+L2_noise^p2))^(1/p2);
    end

    [phi2,lambda2]=eigs(Lp2,k,'sm');
    [phi2_noise,lambda2_noise]=eigs(Lp2_noise,k,'sm');
    
    %%
    p3=-20;
    delta=log(1+abs(p3));
    if(p3==1)
        L1=eye(n)-D1invsqrt*W1*D1invsqrt;
        L2=eye(n)-D2invsqrt*W2*D2invsqrt;
        L2_noise=eye(n)-D2_noise_invsqrt*W2_noise*D2_noise_invsqrt;
        Lp3=(1/T)*(L1+L2);
        Lp3_noise=(1/T)*(L1+L2_noise);
    elseif(p3>0)
        L1=eye(n)-D1invsqrt*W1*D1invsqrt;
        L2=eye(n)-D2invsqrt*W2*D2invsqrt;
        L2_noise=eye(n)-D2_noise_invsqrt*W2_noise*D2_noise_invsqrt;
        Lp3=((1/T)*(L1^p3+L2^p3))^(1/p3);
        Lp3_noise=((1/T)*(L1^p3+L2_noise^p3))^(1/p3);
    else
        L1=(1+delta)*eye(n)-D1invsqrt*W1*D1invsqrt;
        L2=(1+delta)*eye(n)-D2invsqrt*W2*D2invsqrt;
        L2_noise=(1+delta)*eye(n)-D2_noise_invsqrt*W2_noise*D2_noise_invsqrt;
        Lp3=((1/T)*(L1^p3+L2^p3))^(1/p3);
        Lp3_noise=((1/T)*(L1^p3+L2_noise^p3))^(1/p3);
    end

    k=2;
    [phi3,lambda3]=eigs(Lp3,k,'sm');
    [phi3_noise,lambda3_noise]=eigs(Lp3_noise,k,'sm');
    
    %%
    p4=-30;
    delta=log(1+abs(p4));
    if(p4==1)
        L1=eye(n)-D1invsqrt*W1*D1invsqrt;
        L2=eye(n)-D2invsqrt*W2*D2invsqrt;
        L2_noise=eye(n)-D2_noise_invsqrt*W2_noise*D2_noise_invsqrt;
        Lp4=(1/T)*(L1+L2);
        Lp4_noise=(1/T)*(L1+L2_noise);
    elseif(p4>0)
        L1=eye(n)-D1invsqrt*W1*D1invsqrt;
        L2=eye(n)-D2invsqrt*W2*D2invsqrt;
        L2_noise=eye(n)-D2_noise_invsqrt*W2_noise*D2_noise_invsqrt;
        Lp4=((1/T)*(L1^p4+L2^p4))^(1/p4);
        Lp4_noise=((1/T)*(L1^p4+L2_noise^p4))^(1/p4);
    else
        L1=(1+delta)*eye(n)-D1invsqrt*W1*D1invsqrt;
        L2=(1+delta)*eye(n)-D2invsqrt*W2*D2invsqrt;
        L2_noise=(1+delta)*eye(n)-D2_noise_invsqrt*W2_noise*D2_noise_invsqrt;
        Lp4=((1/T)*(L1^p4+L2^p4))^(1/p4);
        Lp4_noise=((1/T)*(L1^p4+L2_noise^p4))^(1/p4);
    end

    k=2;
    [phi4,lambda4]=eigs(Lp4,k,'sm');
    [phi4_noise,lambda4_noise]=eigs(Lp4_noise,k,'sm');

    %% Allen-Cahn
    % Allen-Cahn-parameter definition
    omega0=1000;
    epsilon=5.0e-03;
    dt=0.01;
    c=omega0+3/epsilon;
    max_iter=300;
    tolit=1.0e-06;

    % iteration
    [U1,~]=convexity_splitting_vector_modified_fast(U0,lambda1,phi1,omega0,epsilon,dt,c,max_iter,tolit);
    [U2,~]=convexity_splitting_vector_modified_fast(U0,lambda2,phi2,omega0,epsilon,dt,c,max_iter,tolit);
    [U3,~]=convexity_splitting_vector_modified_fast(U0,lambda3,phi3,omega0,epsilon,dt,c,max_iter,tolit);
    [U4,~]=convexity_splitting_vector_modified_fast(U0,lambda4,phi4,omega0,epsilon,dt,c,max_iter,tolit);
    [U1_noise,~]=convexity_splitting_vector_modified_fast(U0,lambda1_noise,phi1_noise,omega0,epsilon,dt,c,max_iter,tolit);
    [U2_noise,~]=convexity_splitting_vector_modified_fast(U0,lambda2_noise,phi2_noise,omega0,epsilon,dt,c,max_iter,tolit);
    [U3_noise,~]=convexity_splitting_vector_modified_fast(U0,lambda3_noise,phi3_noise,omega0,epsilon,dt,c,max_iter,tolit);
    [U4_noise,~]=convexity_splitting_vector_modified_fast(U0,lambda4_noise,phi4_noise,omega0,epsilon,dt,c,max_iter,tolit);
    
    % retrieve solution and calculate accuracy
    U_sol1=zeros(n,m);
    [~, I_U1] = max(U1,[],2);
    for i=1:n
        U_sol1(i,I_U1(i))=1;
    end
    accuracy_PML_p_pos(j)=sum(all(U_sol1==Y_mat,2))/n;

    U_sol2=zeros(n,m);
    [~, I_U2] = max(U2,[],2);
    for i=1:n
        U_sol2(i,I_U2(i))=1;
    end
    accuracy_PML_p_1(j)=sum(all(U_sol2==Y_mat,2))/n;
    
    U_sol3=zeros(n,m);
    [~, I_U3] = max(U3,[],2);
    for i=1:n
        U_sol3(i,I_U3(i))=1;
    end
    accuracy_PML_p_minus_20(j)=sum(all(U_sol3==Y_mat,2))/n;
        
    U_sol4=zeros(n,m);
    [~, I_U4] = max(U4,[],2);
    for i=1:n
        U_sol4(i,I_U4(i))=1;
    end
    accuracy_PML_p_minus_30(j)=sum(all(U_sol4==Y_mat,2))/n;

    U_sol1_noise=zeros(n,m);
    [~, I_U1_noise] = max(U1_noise,[],2);
    for i=1:n
        U_sol1_noise(i,I_U1_noise(i))=1;
    end
    accuracy_PML_p_pos_noise(j)=sum(all(U_sol1_noise==Y_mat,2))/n;

    U_sol2_noise=zeros(n,m);
    [~, I_U2_noise] = max(U2_noise,[],2);
    for i=1:n
        U_sol2_noise(i,I_U2_noise(i))=1;
    end
    accuracy_PML_p_1_noise(j)=sum(all(U_sol2_noise==Y_mat,2))/n;
    
    U_sol3_noise=zeros(n,m);
    [~, I_U3_noise] = max(U3_noise,[],2);
    for i=1:n
        U_sol3_noise(i,I_U3_noise(i))=1;
    end
    accuracy_PML_p_minus_20_noise(j)=sum(all(U_sol3_noise==Y_mat,2))/n;
    
    U_sol4_noise=zeros(n,m);
    [~, I_U4_noise] = max(U4_noise,[],2);
    for i=1:n
        U_sol4_noise(i,I_U4_noise(i))=1;
    end
    accuracy_PML_p_minus_30_noise(j)=sum(all(U_sol4_noise==Y_mat,2))/n;
end

%% print classification error
fprintf('informative layers: \n    p   |  10  |   1  | -20  | -30  \n')
fprintf('  error | %4.2f | %4.2f | %4.2f | %4.2f \n', [(1-mean(accuracy_PML_p_pos))*100,(1-mean(accuracy_PML_p_1))*100,(1-mean(accuracy_PML_p_minus_20))*100,(1-mean(accuracy_PML_p_minus_30))*100])
fprintf('one noisy layer: \n    p   |  10  |   1  | -20  | -30 \n')
fprintf('  error | %4.1f | %4.1f | %4.1f | %4.1f \n', [(1-mean(accuracy_PML_p_pos_noise))*100,(1-mean(accuracy_PML_p_1_noise))*100,(1-mean(accuracy_PML_p_minus_20_noise))*100,(1-mean(accuracy_PML_p_minus_30_noise))*100])
