% Numerical experiment on 3-layer stochastic block model (SBM) graphs [2]
% with each layer carrying the information about one class
% for the Allen-Cahn multiclass classification scheme [1, Algorithm 6.1]
%
% using the power mean Laplacian [3] with two and three layers as well as the 
% three single layer graphs. Table [1, Fig. 3] shows results for varying powers p.
% Data generated by the routine generate_sbm_graph [4].
%
% using the power mean Laplacian [3] with two and three layers as well as
% the three single layer graphs. Table [1, Fig. 3] shows results for 
% varying powers p. Data generated by the routine generate_sbm_graph [4].
%
%
% [1] Kai Bergermann, Martin Stoll, and Toni Volkmer. Semi-supervised Learning for Multilayer Graphs Using Diffuse Interface Methods and Fast Matrix Vector Products. Submitted, 2020. 
% [2] P. W. Holland, K. B. Laskey, and S. Leinhardt, Stochastic blockmodels: First steps, Social networks, 5 (1983), pp. 109–137.
% [3] Pedro Mercado, Antoine Gautier, Francesco Tudisco, and Matthias Hein. The power mean Laplacian for multilayer graph clustering. In Proceedings of the Twenty-First International Conference on Artificial Intelligence and Statistics, volume 84 of Proceedings of Machine Learning Research, pages 1828-1838, 2018.
% [4] P. Mercado, F. Tudisco, and M. Hein, Generalized matrix means for semi-supervised learning with multilayer graphs, 2019.
%
% This software is distributed under the GNU General Public License v2. See 
% COPYING for the full license text. If that file is not available, see 
% <http://www.gnu.org/licenses/>.
%
% Copyright (c) 2020 Kai Bergermann

clear all
addpath('../Subroutines')

%% create label information
n_cluster=50;
m=3; T=3;
n=n_cluster*m;

fprintf('  p  |  L1  |  L2  |  L3  | Lp_12 | Lp_13 | Lp_23 |  Lp \n')

for p = [10 5 1 -1 -5 -10 -20 -30 -50]
    fprintf('--------------------------------------------------------\n');

    clusters1=[ones(n_cluster,1);2*ones(2*n_cluster,1)];
    clusters2=[2*ones(n_cluster,1);ones(n_cluster,1);2*ones(n_cluster,1)];
    clusters3=[2*ones(2*n_cluster,1);ones(n_cluster,1)];
    
    %% create true label matrix
    Y_mat=[ones(n_cluster,1),zeros(n_cluster,2);
        zeros(n_cluster,1), ones(n_cluster,1), zeros(n_cluster,1);
        zeros(n_cluster,2), ones(n_cluster,1)];
    
    %% create initial conditions U0
    U0=Y_mat;
    perc_damaged=0.96;
    for i=1:n_cluster
        if(i<=n_cluster*perc_damaged)
            U0(i,:)=(1/m)*ones(m,1);
            U0(i+n_cluster,:)=(1/m)*ones(m,1);
            U0(i+2*n_cluster,:)=(1/m)*ones(m,1);
        end
    end
    
    %% SBM, generate weight matrices
    p_in=.7; p_out=.3;
    n_graphs=100;
    accuracy_L1=zeros(3*n_graphs,1);
    accuracy_L2=zeros(3*n_graphs,1);
    accuracy_L3=zeros(3*n_graphs,1);
    accuracy_Lp_2layer_12=zeros((3/2)*n_graphs,1);
    accuracy_Lp_2layer_13=zeros((3/2)*n_graphs,1);
    accuracy_Lp_2layer_23=zeros((3/2)*n_graphs,1);
    accuracy_Lp=zeros(n_graphs,1);
    rng('default');
    rng(2406);

    avgTimeSBM = 0;
    tTotal = tic;
    %% L, 1 layer (3 combinations)
	for j=1:3*n_graphs
        tSBM = tic;
        [W1,~]=generate_sbm_graph(clusters1, p_in, p_out);
        [W2,~]=generate_sbm_graph(clusters2, p_in, p_out);
        [W3,~]=generate_sbm_graph(clusters3, p_in, p_out);
        avgTimeSBM = avgTimeSBM + toc(tSBM);
        
        %% Plot sparsity pattern of the weight matrices
        subplot(131)
        spy(W1)
        subplot(132)
        spy(W2)
        subplot(133)
        spy(W3)
        
        %% Create single layer graph Laplacians
        D1invsqrt=diag(sum(W1).^-0.5);
        D2invsqrt=diag(sum(W2).^-0.5);
        D3invsqrt=diag(sum(W3).^-0.5);
        
        delta=log(1+abs(p));
        if(p==1)
            L1=eye(n)-D1invsqrt*W1*D1invsqrt;
            L2=eye(n)-D2invsqrt*W2*D2invsqrt;
            L3=eye(n)-D3invsqrt*W3*D3invsqrt;
        elseif(p>0)
            L1=eye(n)-D1invsqrt*W1*D1invsqrt;
            L2=eye(n)-D2invsqrt*W2*D2invsqrt;
            L3=eye(n)-D3invsqrt*W3*D3invsqrt;
        else
            L1=(1+delta)*eye(n)-D1invsqrt*W1*D1invsqrt;
            L2=(1+delta)*eye(n)-D2invsqrt*W2*D2invsqrt;
            L3=(1+delta)*eye(n)-D3invsqrt*W3*D3invsqrt;
        end
        
        % compute eigenpairs
        k=3;
        [phi_L1,lambda_L1]=eigs(L1,k,-1); % 'sm' may cause warning/cause error here
        [phi_L2,lambda_L2]=eigs(L2,k,-1); % 'sm' may cause warning/cause error here
        [phi_L3,lambda_L3]=eigs(L3,k,-1); % 'sm' may cause warning/cause error here
        
        %% Allen-Cahn
        % Allen-Cahn-parameter definition
        omega0=1000;
        epsilon=5.0e-03;
        dt=0.01;
        c=omega0+3/epsilon;
        max_iter=300;
        tolit=1.0e-06;
        
        % iteration
        [U_L1,~]=convexity_splitting_vector_modified_fast(U0,lambda_L1,phi_L1,omega0,epsilon,dt,c,max_iter,tolit);
        [U_L2,~]=convexity_splitting_vector_modified_fast(U0,lambda_L2,phi_L2,omega0,epsilon,dt,c,max_iter,tolit);
        [U_L3,~]=convexity_splitting_vector_modified_fast(U0,lambda_L3,phi_L3,omega0,epsilon,dt,c,max_iter,tolit);
        
        % retrieve solution and calculate accuracy
        U_sol_L1=zeros(n,m);
        [~, I_L1] = max(U_L1,[],2);
        for i=1:n
            U_sol_L1(i,I_L1(i)) = 1;
        end
        accuracy_L1(j)=sum(all(U_sol_L1==Y_mat,2))/n;
        
        U_sol_L2=zeros(n,m);
        [~, I_L2] = max(U_L2,[],2);
        for i=1:n
            U_sol_L2(i,I_L2(i)) = 1;
        end
        accuracy_L2(j)=sum(all(U_sol_L2==Y_mat,2))/n;
        
        U_sol_L3=zeros(n,m);
        [~, I_L3] = max(U_L3,[],2);
        for i=1:n
            U_sol_L3(i,I_L3(i)) = 1;
        end
        accuracy_L3(j)=sum(all(U_sol_L3==Y_mat,2))/n;
    end
    avgTimeTotal = toc(tTotal) / (3*n_graphs) / 3;
    avgTimeSBM = avgTimeSBM / (3*n_graphs) / 3;

    avgTimeSBM2 = 0;
    tTotal = tic;
    %% Lp, 2 layers (3 combinations)
	for j=1:(3/2)*n_graphs
        tSBM = tic;
        [W1,~]=generate_sbm_graph(clusters1, p_in, p_out);
        [W2,~]=generate_sbm_graph(clusters2, p_in, p_out);
        [W3,~]=generate_sbm_graph(clusters3, p_in, p_out);
        avgTimeSBM2 = avgTimeSBM2 + toc(tSBM);
        
        %% Plot sparsity pattern of the weight matrices
        subplot(131)
        spy(W1)
        subplot(132)
        spy(W2)
        subplot(133)
        spy(W3)
        
        %% Create single layer graph Laplacians
        D1invsqrt=diag(sum(W1).^-0.5);
        D2invsqrt=diag(sum(W2).^-0.5);
        D3invsqrt=diag(sum(W3).^-0.5);
        
        delta=log(1+abs(p));
        if(p==1)
            L1=eye(n)-D1invsqrt*W1*D1invsqrt;
            L2=eye(n)-D2invsqrt*W2*D2invsqrt;
            L3=eye(n)-D3invsqrt*W3*D3invsqrt;
            Lp_2layer_12=(1/2)*(L1+L2);
            Lp_2layer_13=(1/2)*(L1+L3);
            Lp_2layer_23=(1/2)*(L2+L3);
        elseif(p>0)
            L1=eye(n)-D1invsqrt*W1*D1invsqrt;
            L2=eye(n)-D2invsqrt*W2*D2invsqrt;
            L3=eye(n)-D3invsqrt*W3*D3invsqrt;
            Lp_2layer_12=((1/2)*(L1^p+L2^p))^(1/p);
            Lp_2layer_13=((1/2)*(L1^p+L3^p))^(1/p);
            Lp_2layer_23=((1/2)*(L2^p+L3^p))^(1/p);
        else
            L1=(1+delta)*eye(n)-D1invsqrt*W1*D1invsqrt;
            L2=(1+delta)*eye(n)-D2invsqrt*W2*D2invsqrt;
            L3=(1+delta)*eye(n)-D3invsqrt*W3*D3invsqrt;
            Lp_2layer_12=((1/2)*(L1^p+L2^p))^(1/p);
            Lp_2layer_13=((1/2)*(L1^p+L3^p))^(1/p);
            Lp_2layer_23=((1/2)*(L2^p+L3^p))^(1/p);
        end
        
        % compute eigenpairs
        k=3;
        [phi_Lp_2layer_12,lambda_Lp_2layer_12]=eigs(Lp_2layer_12,k,'sm');
        [phi_Lp_2layer_13,lambda_Lp_2layer_13]=eigs(Lp_2layer_13,k,'sm');
        [phi_Lp_2layer_23,lambda_Lp_2layer_23]=eigs(Lp_2layer_23,k,'sm');
        
        %% Allen-Cahn
        % Allen-Cahn-parameter definition
        omega0=1000;
        epsilon=5.0e-03;
        dt=0.01;
        c=omega0+3/epsilon;
        max_iter=300;
        tolit=1.0e-06;
        
        % iteration
        [U_Lp_2layer_12,~]=convexity_splitting_vector_modified_fast(U0,lambda_Lp_2layer_12,phi_Lp_2layer_12,omega0,epsilon,dt,c,max_iter,tolit);
        [U_Lp_2layer_13,~]=convexity_splitting_vector_modified_fast(U0,lambda_Lp_2layer_13,phi_Lp_2layer_13,omega0,epsilon,dt,c,max_iter,tolit);
        [U_Lp_2layer_23,~]=convexity_splitting_vector_modified_fast(U0,lambda_Lp_2layer_23,phi_Lp_2layer_23,omega0,epsilon,dt,c,max_iter,tolit);
        
        % retrieve solution and calculate accuracy
        U_sol_Lp_2layer_12=zeros(n,m);
        [~, I_L12] = max(U_Lp_2layer_12,[],2);
        for i=1:n
            U_sol_Lp_2layer_12(i,I_L12(i)) = 1;
        end
        accuracy_Lp_2layer_12(j)=sum(all(U_sol_Lp_2layer_12==Y_mat,2))/n;
        
        U_sol_Lp_2layer_13=zeros(n,m);
        [~, I_L13] = max(U_Lp_2layer_13,[],2);
        for i=1:n
            U_sol_Lp_2layer_13(i,I_L13(i)) = 1;
        end
        accuracy_Lp_2layer_13(j)=sum(all(U_sol_Lp_2layer_13==Y_mat,2))/n;
        
        U_sol_Lp_2layer_23=zeros(n,m);
        [~, I_L23] = max(U_Lp_2layer_23,[],2);
        for i=1:n
            U_sol_Lp_2layer_23(i,I_L23(i)) = 1;
        end
        accuracy_Lp_2layer_23(j)=sum(all(U_sol_Lp_2layer_23==Y_mat,2))/n;
    end
    avgTimeTotal2 = toc(tTotal) / ((3/2)*n_graphs) / 3;
    avgTimeSBM2 = avgTimeSBM2 / ((3/2)*n_graphs) / 3;

    avgTimeSBM3 = 0;
    tTotal = tic;
    %% Lp, 3 layers
    for j=1:n_graphs
        tSBM = tic;
        [W1,~]=generate_sbm_graph(clusters1, p_in, p_out);
        [W2,~]=generate_sbm_graph(clusters2, p_in, p_out);
        [W3,~]=generate_sbm_graph(clusters3, p_in, p_out);
        avgTimeSBM3 = avgTimeSBM3 + toc(tSBM);
        
        %% Create single layer graph Laplacians
        D1invsqrt=diag(sum(W1).^-0.5);
        D2invsqrt=diag(sum(W2).^-0.5);
        D3invsqrt=diag(sum(W3).^-0.5);
        
        delta=log(1+abs(p));
        if(p==1)
            L1=eye(n)-D1invsqrt*W1*D1invsqrt;
            L2=eye(n)-D2invsqrt*W2*D2invsqrt;
            L3=eye(n)-D3invsqrt*W3*D3invsqrt;
            Lp=(1/T)*(L1+L2+L3);
        elseif(p>0)
            L1=eye(n)-D1invsqrt*W1*D1invsqrt;
            L2=eye(n)-D2invsqrt*W2*D2invsqrt;
            L3=eye(n)-D3invsqrt*W3*D3invsqrt;
            Lp=((1/T)*(L1^p+L2^p+L3^p))^(1/p);
        else
            L1=(1+delta)*eye(n)-D1invsqrt*W1*D1invsqrt;
            L2=(1+delta)*eye(n)-D2invsqrt*W2*D2invsqrt;
            L3=(1+delta)*eye(n)-D3invsqrt*W3*D3invsqrt;
            Lp=((1/T)*(L1^p+L2^p+L3^p))^(1/p);
        end
        
        % compute eigenpairs
        k=3;
        [phi_Lp,lambda_Lp]=eigs(Lp,k,'sm');
        
        %% Allen-Cahn
        % Allen-Cahn-parameter definition
        omega0=1000;
        epsilon=5.0e-03;
        dt=0.01;
        c=omega0+3/epsilon;
        max_iter=300;
        tolit=1.0e-06;
        
        % iteration
        [U_Lp,~]=convexity_splitting_vector_modified_fast(U0,lambda_Lp,phi_Lp,omega0,epsilon,dt,c,max_iter,tolit);
        
        % retrieve solution and calculate accuracy
        U_sol_Lp=zeros(n,m);
        [~, I_Lp] = max(U_Lp,[],2);
        for i=1:n
            U_sol_Lp(i,I_Lp(i)) = 1;
        end
        accuracy_Lp(j)=sum(all(U_sol_Lp==Y_mat,2))/n;
        
    end
    avgTimeTotal3 = toc(tTotal) / n_graphs;
    avgTimeSBM3 = avgTimeSBM3 / n_graphs;
              
    %% print classification error
    fprintf('%4.0f | %4.1f | %4.1f | %4.1f |  %4.1f |  %4.1f |  %4.1f | %4.2f \n', [p, (1-mean(accuracy_L1))*100, (1-mean(accuracy_L2))*100, (1-mean(accuracy_L3))*100, (1-mean(accuracy_Lp_2layer_12))*100, (1-mean(accuracy_Lp_2layer_13))*100, (1-mean(accuracy_Lp_2layer_23))*100, (1-mean(accuracy_Lp))*100]);
    fprintf('avg. SBM construction time per graph: 1 layer: %.3f, 2 layer: %.3f, 3 layer: %.3f\n', avgTimeSBM, avgTimeSBM2, avgTimeSBM3);
    fprintf('avg. total time per graph: 1 layer: %.3f, 2 layer: %.3f, 3 layer: %.3f\n', avgTimeTotal, avgTimeTotal, avgTimeTotal);
end
