%% / Create parameter structure for microcircuit model of attention
% @author Frederik Beuth
%
% In the first part, this file contains parameters which the user is free to change. The remaining contains fixed parameters which should not be changed.
% Notation of the parameters:
% - w* => Weight pattern
% - v* => Single value to weight the influence of the whole pattern
% - *Sup* => Suppressive weight pattern or value
%
%\
%> @param L Number of feature in layer 4 and layer 2/3
%> @param X Number of locations in layer 4
%> @param rfSizeL2 Size of the receptive fields in layer 2/3. Size is denoted in number of presynaptic neurons
%> @param useLargeRfEnvelope If true, double the envelope size of the receptive field Gaussian. Important, the width of the Gaussian and so  
%>                            the receptive field size is unchanged.
function this = initModel(L, X, rfSizeL2, useLargeRfEnvelope)

   %% Variable model parameters  
   % Parameter controlling the contrast response function (CRF)
   % - Calibrated according to Fig. 6
   % - sigmaL4: first contrast enhancement parameter, lower values shift CRF to the left 
   % - vP: steepness of the CRF function (higher values increase steepness)
   % - vE: second contrast enhancement parameter, lower values shift CRF to the left      
   this.sigmaL4 = 0.3;   
   this.vE = 3.0;        
   this.pE = 2.0;      
   
   % Influence of each connection
   this.vSP = 3;         % Spatial amplification from FEF/LIP to layer 4
   this.vFeatL4 = 1;     % Amplification from layer 2/3 to layer 4
   this.vFeatL2 = 0.5;   % Feature-based amplification from PFC to layer 2/3
   this.vSupFeat = 3;    % Feature-based suppression from layer 2/3 to layer 4 
   this.pSupFeat = 2;    % Feature-based suppression from layer 2/3 to layer 4, amount of non-linearity
   this.vSupSur = 0.5;   % Surround suppression from layer 2/3 to layer 4 
   this.pSupSur = 1;     % Surround suppression from layer 2/3 to layer 4, amount of non-linearity 
   
   % Baseline activity (beta) of layer 4 and layer 2/3. This amount is not modulated by attention and suppression is
   % not induced by it
   this.baseline = 0;
   
   % Non-linearity parameter for layer 2/3 pooling operation
   this.poolingP1 = 4;


   %% Fixed parameters (do not change them!)      
   this.sigmaL2 = 2;     % This high value do mostly no contrast enhancement, but calculate response values 
                         % to a meaningful range               
  
   % Timing
   this.tau = 10;
   this.tEnd = 100;   % simulation time
   this.tPlot = this.tEnd;
   assert(this.tPlot<=this.tEnd);
      
   % Step size for the Euler method to solve the ODEs
   this.hL4 = 1;
   this.hL2 = 1;   

   % Security checks
   if(L<2)
      error('Feature space to small, need at least 2 neurons');
   end
   if(X<3)
      error('Location space to small, need at least 3 neurons');
   end
   if(X<rfSizeL2)
      error('Location space to small, must be at least the size of the receptive field');
   end

   % Layer resolutions (part 1)
   this.resV1 = [L,X];
   this.resL4 = [L,X];
   
   %% Receptive field (rf) definitions for layer 2/3  
   % - The receptive field is represented by neuronal connectivity weights in our model.
   % - Receptive field borders are defined by the location at which the cell reacts with 50% max response. We define for this purpose
   %   the weights so that they have a value of 0.5 at the border, via a Gaussian with an appropriate standard deviation: sigma = size / 2.5 .
   % - The rf size is defined as the distance between the borders, specified via parameter rfSizeL2.
   % - The rf overlaps to 50% with other receptive fields   
   this.rfSize_FF = rfSizeL2;    
   
   % Parameter and discretization of the Gaussian
   % Option 1 (Standard): Discretized the Gaussian only inside the rf.
   % - Envelope size is equal to rf size.
   % - Standard deviation is chosen so that receptive field border induces 50% response.
   % - Overlap between receptive fields is 50%.
   if(nargin<4)
      useLargeRfEnvelope = false;
   end
   if(~useLargeRfEnvelope)
      this.rfEnvelop_FF = rfSizeL2; % Size of the weight matrix
      rfOverlap = 0.5;           % Overlap between weight matrices
      rfSigma = 2.5;             % Standard deviation = receptive field size / rfSigma
   else
      % Option 2: Discretized the Gaussian within a envelope of two-times the receptive field
      % - The Gaussian is discretized almost fully within this large envelope.
      % - Gaussian envelope is twice the size of option 1.
      % - Thus, the overlap between weight matrices has to be adapted from 50% to 25%.
      % - Sigma is the same as in option 1.
      % - The resulting resolution of layer 2/3 is similar as in option 1.
      this.rfEnvelop_FF = (rfSizeL2-1)*2 + 1;
      rfOverlap = 0.25;
      rfSigma = 2.5;
   end
   
   

   %% Feedforward (FF) processing (layer 4 to layer 2/3)
   % Receptive field indices
   this.rfshift_FF = (this.rfEnvelop_FF-1)*rfOverlap; 
   this.rfH_FF = (this.rfEnvelop_FF-1)/2;             % rfH => receptive field envelope half size   
   this.rfX_FF = round(0.001+[1 : this.rfshift_FF : this.resL4(2); ...
      this.rfEnvelop_FF : this.rfshift_FF : this.resL4(2)+this.rfEnvelop_FF-1]); % start and end indices of the rf envelope
   this.rfX_FF(3,:) = mean(this.rfX_FF,1);            % Center of each rf
   % Gaussian kernel for weights
   % - Sigma is chosen so that the response is about 50% at the rf border 
   this.wFF = gauss1D(this.rfEnvelop_FF, (this.rfEnvelop_FF+1)/2, this.rfSize_FF/rfSigma); 
   this.wFF_2D = repmat(this.wFF, [L,1]);

   % Layer resolutions (part 2)
   this.resL2 = [L,size(this.rfX_FF,2)];
   
   
   %% Feedback (FB) processing (layer 2/3 to layer 4)
   % - Connectivity is chosen reciprocal to feedforward processing
   % - For this, the sourcecode below parses the receptive fields of the feedforward stream
   % - RF envelope is individual for each location to handle discretization and border issues
   % - By this, it is guaranteed that a L4 neuron receives feedback only from L2 neurons which are driven by it.
   %   Without, asymmetries can occur in biased competition as well as activity can be spatially shifted or skewed  
   %
   % - The sigma of the Gaussian is identical for all locations, defining the spatial extend of the feedback
   % - This guarantees that all L4 neurons have feedback with the same spatial extend, 
   %   even if the presynaptic number of L2 neurons differ.
   % - Thus, the Gaussian might be differently discretized
   % - Furthermore, the rf centers are determined directly and not rounded to full numbers, so the receptive field center 
   %   is always aligned precisely at the position of the postsynaptic neuron

   % Parse receptive fields of FF rf to determine envelope borders of FB rf
   this.rfX_FB = zeros(2, this.resL4(2));
   for x=1:this.resL4(2)
      this.rfX_FB(1,x) = find(this.rfX_FF(2,:) >= x+this.rfH_FF, 1, 'first');
      this.rfX_FB(2,x) = find(this.rfX_FF(1,:) <= x+this.rfH_FF, 1, 'last');
   end
   % Center coordinates of rf, not rounded to avoid asymmetries 
   % - Thus, the rf is centered at the postsynaptic neuron location
   this.rfX_FB(3,:) = linspace(1 , this.resL2(2), this.resL4(2)); % center indices of each rf
      
   % Determine spatial extend of feedback receptive field from size ratio between L2 and L4, plus 
   % from feedforward receptive field size
   this.rfSize_FB = this.resL2(2)./this.resL4(2) * this.rfSize_FF;
   
   % Envelope of feedback connectivity
   this.envelope_FB = this.rfX_FB(2,:) - this.rfX_FB(1,:) + 1;   
   this.rfH_FB = (max(this.envelope_FB)-1)/2;           % rfH => receptive field envelope half size     
   
   % Coordinates are in extended layer coordinates, so add an extension
   this.rfX_FB = this.rfX_FB + this.rfH_FB;
   
   % Create Gaussian feedback weights
   this.wFB = cell(this.resL4(2));
   this.wFB_2D = cell(this.resL4(2));   
   for x=1:this.resL4(2)
      mu = this.rfX_FB(3,x)-this.rfX_FB(1,x)+1; % center of Gaussian relative in the envelope
      this.wFB{x} = gauss1D(this.envelope_FB(x), mu, this.rfSize_FB/rfSigma); 
      this.wFB_2D{x} = repmat(this.wFB{x}, [L,1]);
   end
   
   
   %% Surround suppression 
   % - Array has for each distance in L2 a suppression weight
   % - Suppression is zero for very close distances, i.e. for all L2 cells driven by a single L4 cell 
   % - Suppression is high for close distances, necessary for the response gain effect as it results from suppression from adjacent neurons
   % - Suppression is average-low for far distances, necessary for surround experiments (Cavanaugh2002a)
   rfH_SUR = round((this.rfSize_FB-1)/2);  % half receptive field size
   
   this.wSupSur = zeros(this.resL2(2)+this.rfH_FB,1);
   this.wSupSur(1:rfH_SUR+1) = linspace(0,1, rfH_SUR+1); 
   this.wSupSur(rfH_SUR+2:2*rfH_SUR+2) = linspace(1,0.4, rfH_SUR+1);
   this.wSupSur(2*rfH_SUR+2:end) = 0.4;
      

   %% Feature based suppression
   % - wSupFeat has for each distance between the features one entry
   % - Feature space is circular, hence suppression is maximal for L/2 ('Lhalf')
   Lzero = floor(L/8);      % Range of zero suppression between close features
   Lhalf = ceil((L-1)/2)+1; 
   
   this.wSupFeat = zeros(L,1);   
   for l=1:L
      if(l<=Lzero)
         this.wSupFeat(l) = 0;
      elseif(l<=Lhalf)
         this.wSupFeat(l) = (l-Lzero-1)/(Lhalf-Lzero-1);
      elseif(l<=L-Lzero)
         this.wSupFeat(l) = 1-(l-Lhalf-1)/(Lhalf-Lzero-1);
      else
         this.wSupFeat(l) = 0;
      end
   end
   this.wSupFeat = this.wSupFeat ./ sum(this.wSupFeat);      
end
