%% / Calculate a 1D Gaussian kernel within a circular space
% @author Frederik Beuth
%
% This revision assumes a circular space, hence 1 and X are adjacent values
%
% Idea:
% - Create first a symetrical and central aligned Gaussian function
% - Secondly, shift the created Gaussian to the requested mean
% - Third, copy the remaining part from the central Gaussian
%
%\
%> @param X Size of the envelope
%> @param mu Center of the Gauss. Valid values are in the range 1:kernelsize 
%> @param sigma Standard deviation
function Y = gauss1Dcircular(X, mu, sigma)

   % Self-demonstration
   if(nargin<1)
      X = 33;
      mu = round((X-1)*0.7+1);
      sigma = X/5;
   end
   
   mu2 = round(mu);   
   Xhalf = (X-1)/2+1;

   if(mu2<1)
      error('Mu (%i) is <1!', mu);
   end
   
   if(mu2>X)
      error('Mu (%i) is >X!', mu);
   end
   
   %create full function
   K0 = gauss1D(X,Xhalf,sigma);
   Y = zeros(X,1);
   
      
   % case 1: mu==xHalf => function is precisely aligned in the space, we have nothing to do
   % case 2: mu<xHalf => right side has to start again
   % case 3: mu>xHalf => left side has to start again   
   if(mu2==Xhalf)
      Y = K0;
      
   elseif(mu2<Xhalf)
      xAgain = mu2 + Xhalf;            % Denote the coordinate to restart the curve. This will be the lowest point in the final curve(Y)
      xRest = Xhalf-mu2;               % Denote the remaining part after the restart
      Y(1:xAgain) = K0(xRest:end);
      Y(xAgain+1:end) = K0(1:xRest-1);
      
   elseif(mu2>Xhalf)
      xAgain = mu2 - Xhalf;
      x0 = X - xAgain + 1;             % Denote the coordinate in the original curve to copy from
      Y(1:xAgain) = K0(x0:end);
      Y(xAgain+1:end) = K0(1:x0-1);
   end  
   
   % Self-demonstration
   if(nargin<1)      
      figure
      plot(Y)
      axis([1, X, 0, 1]);
   end
   
end
   
   
   
   
