%% / Experimental setup for attentional modulation of surround suppression
% @author Frederik Beuth
%
% Data: Sundberg, Mitchell, Reynolds, 2009, Fig. 6F
%
%% @section X14b1 Significant results (p. 957, left column):
% - In attend away condition, surround suppression is weakest for the highest contrast and becomes stronger for lower ones.
% - Attention to the center reduces surround suppression, whereby attention to the surround increases it. The effect is visible at
%   all contrasts.
% - The outlined results are prominent, but no significance test was executed. 
%
%% @section X14b2 Setup:
%   -  Use primary spatial amplification as we assume that attention comes from tracking the object.
%      Feature-based attention could also be assumed as a certain object is attended. However, 
%      it will not work because all objects are identical and could not be distinguished over their features.
%      => We model it with 90% spatial and 10% feature-based attention as this results in the best fits to 
%      the data.
%   - Spatial attention has the same size as stimulus
%   - Contrast of center stimulus is variable
%   - Surround stimulus has always highest contrast
%   - Stimuli have a size of 2°, eccentricity is unknown. However, this size is very small compared to V4 rf sizes at all eccentricities, 
%     so we assume that a stimulus falls only in a single V4 receptive field.
%
%% @section X14b3 Calibration of the fit:
%   - 1) Fit contrast response parameters (sigma, vE, pE)
%   - 2) Calibrate amount of surround suppression to fit attend away condition. Use pSUR>=2 to get a notable difference between attend-away and
%   attend-surround
%   - 3) Attention to fit attend surround. Attend center is mostly correct as the response will probably saturate
%
%\
%> @param setupDir      Directory with experimental setup, legend figures and a mat-file containing all plotting parameters
%> @param debugOn       If true, display additional information for fitting the data
%> @param stdOn         If true, run model with standard parameters
function figure9b(setupDir, debugOn, stdOn)

   if(nargin<1)
      setupDir = 'setups/';
      debugOn = false;
      stdOn = false;
   end
   load('setups/plotParams.mat');
   sizeFig = [0.5, 0.5, 18, 13.5];
   clear 'plotPositions';
   plotPositions{1} = [0.1, 0.25, 0.25, 0.35*2];
   plotPositions{2} = [0.4, 0.25, 0.25, 0.35*2];
   plotPositions{3} = [0.7, 0.25, 0.25, 0.35*2];
      
  
   %% Run Model
   if(~stdOn)
      modelFile = sprintf('%s/modelSundberg2009_Fig6F.mat', modelDir);
   else
      modelFile = sprintf('%s/modelSundberg2009_Fig6F.mat', modelDirStd);
   end   
   % Conditions:
   % - 1 => center stimulus alone with attend away
   % - 2 => center stimulus alone with attend surround
   % - 3 => center stimulus alone with attend center
   % - 4 => pair with attend away
   % - 5 => pair with attend surround
   % - 6 => pair with attend center      
   C = 6;             % # Conditions
   N = 8;             % # Subexperiments/contrasts
   % The authors stated that the response of neurons were not saturated at highest contrast,
   % so we assume a low maximum contrast value, arbitrary chosen
   maxContrast = 0.25;
   contrasts = [0, 0.035, 0.065, 0.11, 0.19, 0.33, 0.57, 0.99]*maxContrast; 
   nGlobal = N*C;     % # All experiments
   L = 13;            % # Different orientations
   X = 53;            % # Different locations
   Xrf = 5;           % Receptive field size
   Xstim = 1;         % Stimulus size
   x1 = 45;           % Center stimulus location in L4
   x2 = 9;            % Surround stimulus location in L4
   x3 = 23;           % Receptive field center in L2      
   x4 = 5;            % Receptive field of surround in L2      
   XH = (Xstim-1)/2;  % Half of stimulus

   %Model parameter (can be changed after creation)
   objModel = initModel(L,X,Xrf);
   if(~stdOn)
      objModel.vE = 1.6;
      objModel.pE = 1.25;
      objModel.vSupSur = 0.55;
      objModel.pSupSur = 2.5;
   end

   allIn = cell(nGlobal,1);
   allL4 = cell(nGlobal,1);
   allL2 = cell(nGlobal,1);
   allFEF = cell(nGlobal,1);
   allPFC = cell(nGlobal,1);  
   rModel = zeros(C,N);      
   % Record surround stimulus for debugging
   rModelSur = zeros(C,N);

   for s1 = 1:N

      % Stimulus preference at location 1 (l1) and 2(l2)
      % preferred (l1) and anti-preferred stimulus (l2) 
      l1 = ((L-1)*0.5)+1;
      l2 = ((L-1)*0.5)+1;

      c = 8/L;
      g = 0.0; 
      r1 = symetricalExpCircle(L, l1, g, 1, c) * contrasts(s1);
      r2 = symetricalExpCircle(L, l2, g, 1, c) * contrasts(N);
      r1 = repmat(r1, [1,Xstim]);
      r2 = repmat(r2, [1,Xstim]);

      for c = 1:C
         s = (s1-1)*C+c; %global experiment index            

         % Set input
         if(c<=3)
            rIn = zeros(L,X);
            rIn(:,x1-XH:x1+XH) = r1;

         else
            rIn = zeros(L,X);
            rIn(:,x1-XH:x1+XH) = r1;  
            rIn(:,x2-XH:x2+XH) = r2;               
         end            
         allIn{s} = rIn;

         % Set amplification
         allFEF{s} = zeros(1,X);
         allPFC{s} = zeros(L,1); 
         if(c==3 || c==6)
            allFEF{s}(x1-2:x1+2) = 0.9; 
            allPFC{s}(l1) = 0.10; 
         end
         if(c==2 || c==5)
            allFEF{s}(x2-2:x2+2) = 0.9; 
            allPFC{s}(l1) = 0.10; 
         end            

         %% Run model calculation
         [rL4, rL2] = calculateModel(allIn{s}, allFEF{s}, allPFC{s}, objModel);
         allL4{s} = rL4;
         allL2{s} = rL2;

         rModel(c,s1) = rL2(l1,x3);
         rModelSur(c,s1) = rL2(l1,x4); 
      end % [conditions]
      fprintf('.');

   end % [experiments]      

   % normalize model response as data responses
   maxData = max(rModel(:));
   minData = min(rModel(:));      
   rModel = (rModel - minData) ./ (maxData - minData);
   if(debugOn)
      fprintf('\nMax. model response = %0.3f; min. response = %0.3f\n', maxData, minData);
      rModelSur = (rModelSur - minData) ./ (maxData - minData);
   end
   save(modelFile, 'rModel', 'allIn', 'allFEF', 'allPFC', 'allL4', 'allL2', 'C', 'N', 'L', 'nGlobal', 'contrasts', ...
      'X', 'Xrf', 'x1', 'x2', 'x3', 'l1', 'l2', 'rModelSur');   
   
   %% Create plots for model
   figure('paperposition', sizeFig);
   for i=1:3
    
      subplot('position', plotPositions{i}(1,:));  
      for j=1:2
         cj=j+1;
         h(j) = semilogx(contrasts, rModel(i+3*(j-1),:), markersLine{cj}, 'linewidth', 3, 'color', colors(cj,:), 'markersize', 8, ...
            'LineSmoothing','on');
         hold on;
      end
      % For debugging, display the amount of suppression in the range [0..200%]
      if(debugOn)
         j=3;
         cj=1;
         h(j) = semilogx(contrasts, 0.5*rModel(i,:)./rModel(i+3,:), markersLine{cj}, 'linewidth', 3, 'color', colors(cj,:), 'markersize', 8, ...
            'LineSmoothing','on');
      end
      formatPlot(setupDir, colors,  markersLine, markers, noSize, labelSize, true, N, contrasts, i);
      uistack(h(1)); % make attend away condition more visible
   
   end % [i]
   
   %In debug mode, saving the figure and plotting the population response is not necessary
   if(debugOn)
      return; 
   end
      
   %% Saving
   fprintf('\nSaving...\n');
   if(~stdOn)
      name = sprintf('%s/fig9b.png', outputdir);
   else
      name = sprintf('%s/fig9b.png', outputdirStd);
   end
   print(name, '-dpng');  
   
end


%% Function to format the plot nicely
function formatPlot(setupDir, colors,  markersLine, markers, noSize, labelSize, isBottom, N, contrasts, i)

   ylim([0,1.05]);
   xlim([min(contrasts),max(contrasts)]);
   box on   
   
   if(i==1)
      ylabel('Response', 'fontsize', labelSize);
      set(gca, 'YTick', [0,1]);
   else
      set(gca, 'YTicklabel', []);   
   end
   set(gca, 'linewidth', 2, 'fontsize', noSize);

   if(isBottom)
      xlabel('Log contrast', 'fontsize', labelSize);      
   else
      set(gca, 'XTicklabel', []);   
   end
end
