%% / Experimental setup for surround suppression without attention
% @author Frederik Beuth
%
% Data: Cavanaugh2002a, Fig. 8A
%
%% @section X14a1 Significant results (p. 2538, left column):
%  - Surround suppression is significant and at mean 38% (Fig. 3)
%  - Neuron has at lower contrasts a larger receptive field (GSF, grating summation field). GSF diameters were for lowest contrast 
%    2.5 times those measured for high contrast. 
%  - It is explained by Fig. 7 and by "At high contrasts, the surround is relatively strong and suppresses weak responses
%    from the flanks of the center mechanism. At low contrasts the surround is relatively weak, and this suppression is relaxed,
%    allowing more of the center to be seen."
%  - Fig. 8 shows that the model with variable center gain and surround gain fits the data at best.
%  - Here, we follow the same idea, the surround is weaker for lower contrasts because:
%    - The suppression depends on the firing rate and thus being weaker for lower contrasts. 
%    - The data is fitted well with using p^SUR > 1, favoring higher firing rates and thus higher contrasts
%    - Similarly, we model the stimulus along the spatial dimension via a Gaussian.
%
%% @section X14a2 Setup:
%  - Stimuli have a size of 0-7° at 5° eccentricity (plus 9% at 5-10°). V1 RF size is 0.6° at 5° ecc. (Smith et al., 2001).
%
%% @section X14a3 Calibration of the fit:
%   - 1) Calibrate contrast response function (CRF)
%   - 2) Calibrate surround suppression for far locations
%
%\
%> @param setupDir      Directory with experimental setup, legend figures and a mat-file containing all plotting parameters
%> @param debugOn       If true, display additional information for fitting the data
%> @param stdOn         If true, run model with standard parameters
function figure9a(setupDir, debugOn, stdOn)

   if(nargin<1)
      setupDir = 'setups/';
      debugOn = false;
      stdOn = false;
   end
   load(sprintf('%s/plotParams.mat', setupDir));
   
   %% Run Model
   if(~stdOn)
      modelFile = sprintf('%s/modelCavanaugh2002a_Fig8A.mat', modelDir);
   else
      modelFile = sprintf('%s/modelCavanaugh2002a_Fig8A.mat', modelDirStd);
   end
   
   % Stimulus diameter sizes
   diametersModel =  [0, 1, 1.4, 2.8, 4.2, 5.6, 7];

   Cmodel = numel(diametersModel);  % # Conditions/stimulus sizes: 7
   N = 5;             % # Subexperiments/contrasts
   contrasts = [0.03, 0.06, 0.12, 0.25, 0.5];
   nGlobal = N*Cmodel;% # All experiments
   L = 9;             % #different orientations
   X = 163;           % #different locations, should be chosen >largest stimulus covering 152 locations. 163 results in a regular rf structure.
   Xrf = 11;          % Receptive field size
   x1 = (X+1)/2;      % Center stimulus location in L4

   %Model parameter (can be changed after creation)
   useLargeRfEnvelope = true;
   objModel = initModel(L,X,Xrf, useLargeRfEnvelope);     
   if(~stdOn)
      % CRF parameters. Chosen so that the spread of the curves without suppression, i.e at 1.5° stimulus diameter, is equally. This matches the data
      % well.
      objModel.vE = 0.5; 
      objModel.pE = 0.75; 

      % Pooling operation
      objModel.poolingP1 = 2;
      
      % Surround suppression parameters
      % - Set p>1 to decrease suppression for lower contrasts
      objModel.pSupSur = 2.0;
      objModel.vSupSur = 0.8;
  end

   x3 = round((objModel.resL2(2)-1)/2+1);         % Receptive field center in L2

   allIn = cell(nGlobal,1);
   allL4 = cell(nGlobal,1);
   allL2 = cell(nGlobal,1);
   allFEF = cell(nGlobal,1);
   allPFC = cell(nGlobal,1);  
   rModel = nan(N,Cmodel);
   surModel = nan(N,Cmodel);
   h3 = 0;

   for s1 = 1:N

      % Stimulus preference 
      l1 = ((L-1)*0.5)+1;

      % Generate input tuning function
      c = 8/L; 
      g = 0.0; % We do not use a baseline as this is the simplest approach to model the input. Alternatively, a baseline of 0.15 can be used 
      % if current V1 data set should be modelled identical to the other V1 data. A baseline of 0.15 would not change the result.
      r1 = symetricalExpCircle(L, l1, g, 1, c);
      r1 = r1*contrasts(s1); 

      for c = 1:Cmodel         
         s = (s1-1)*Cmodel+c; %global experiment index            

         % Set input
         % - We only have a 1D model in the spatial dimension, whereby the data represents the case of a 2D stimulus with 2D suppression.
         %   Thus, we have to map this 2D stimulus to a 1D input activity: we construct an input activity with spatial extension of the stimulus 
         %   area instead of the diameter.
         rIn = zeros(L,X);
         r = diametersModel(c)/2;               % Stimulus radius 
         cellsPerArea = 4;                      % How many cells are used to represent 1 square millimeter of the stimulus, 
                                                %   i.e a squared stimulus of 1mm x 1mm.
         area = round(cellsPerArea*pi * r^2);   % Stimulus size in L4 cells            
         if(debugOn)
            fprintf('Area=%0.2f\n', area);
         end

         % Create spatial input function
         if(area>0)
            % Gaussian function to represent the spatial dimension
            % - The data can be fitted with both functions, i.e. with an exponential or a Gaussian one. We use a Gaussian function as in 
            %   Cavanaugh2002.
            sigmaSpatial = area/3;
            sp1 = gauss1Dcircular(X,x1, sigmaSpatial);

            rIn = conv2(r1, sp1);
         end            
         allIn{s} = rIn;

         % Set amplifications
         allFEF{s} = zeros(1,X);
         allPFC{s} = zeros(L,1); 

         %% Run model calculation
         [rL4, rL2, supFeat2, supSpatial2] = calculateModel(allIn{s}, allFEF{s}, allPFC{s}, objModel);
         allL4{s} = rL4;
         allL2{s} = rL2;

         rModel(s1,c) = rL2(l1,x3);
         surModel(s1,c) = supSpatial2(l1,x3);
         
         % Printout complete spatial tuning for low, medium, and high contrasts; and small over medium to large stimuli.
         if(debugOn && (s1 == 1 || s1==3 || s1==5) && (c==2 || c==3 || c==4 || c==5 || c==6) )
            if(h3 == 0)
               h3 = figure;
               xsub = 1;
            end
            set(0, 'currentfigure', h3);
            subplot(3,5,xsub);
            rL2_B = imresize(rL2, objModel.resL4, 'lanczos3');
            plot([rIn(l1,:)', rL4(l1,:)', rL2_B(l1,:)', 0.1*supSpatial2(l1,:)'], 'linewidth', 3);
            title(sprintf('Radius=%0.2f, Contrast=%0.2f', r, contrasts(s1)));
            ylim([0,1]);
            if(xsub==1)
               legend({'Input', 'L4', 'L2', 'Surround sup.'});
            end
            xsub = xsub + 1;
         end

      end % [conditions]
      if(~debugOn)
         fprintf('.');
      end

   end % [experiments]     

   % Normalize model responses like data responses
   maxData = max(rModel(:));
   minData = 0;
   if(debugOn)
      fprintf('\nModel max=%02f, min = %02f\n', maxData, minData);
   end
   rModel = (rModel - minData) ./ (maxData - minData);

   save(modelFile, 'rModel', 'allIn', 'allFEF', 'allPFC', 'allL4', 'allL2', 'Cmodel', 'N', 'L', 'nGlobal', 'contrasts', ...
      'X', 'Xrf', 'x1', 'x3', 'l1', 'diametersModel');
   fprintf('\n');
   

   %% Create plot for model
   figure
   for j=N:-1:1
      plot(diametersModel, rModel(j,:), markersLine{j}, 'linewidth', 0.5+0.5*j^1.5, 'color', colors(1,:), 'markersize', 8, ...
         'LineSmoothing','on');
      hold on;
   end
   formatPlot(setupDir, colors,  markersLine, markers, noSize, labelSize, true, N, diametersModel, contrasts);
   
   %In debug mode, saving the figure is not necessary
   if(debugOn)
      return; 
   end
   
   %% Saving
   fprintf('Saving...\n');
   if(~stdOn)
      name = sprintf('%s/fig9a.png', outputdir);
   else
      name = sprintf('%s/fig9a.png', outputdirStd);
   end
   print(name, '-dpng');
end


%% Function to format the plot nicely
function formatPlot(setupDir, colors,  markersLine, markers, noSize, labelSize, isBottom, N, diameters, contrasts)

   ylim([0,1.05]);
   xlim([min(diameters),max(diameters)]);
   box on   
   
   ylabel('Response', 'fontsize', labelSize);
   set(gca, 'YTick', [0,1]);
   set(gca, 'linewidth', 2, 'fontsize', noSize);

   if(isBottom)
      xlabel('Diameter (deg)', 'fontsize', labelSize);      
   end   
end
