%% / Experimental setup for tuning curves in biased competition with feature-based attention
% @author Frederik Beuth
%
% Data: Treue and Martinez, 1999, Fig. 3B
%
%% @section X13b1 Significant results (p. 577, left column):
%  - Significance test were not reported, probably due to lack of space in a Nature paper.
%  - Attention has a strong effect, it increases in average the response to 156%.
%  - No sharpening was observed.
%
%% @section X13b2 Setup:
%  - Stimuli size is unknown, except that two stimuli fit in the RF size of area MT.
%    => We model them as small stimuli compared to the RF size.
%
%% @section X13b3 Calibration of the fit:
%   - Suppression results primarily from the distractor, hence amount of suppression in all 3 conditions
%     results from feature-based suppression.
%   - Thus the suppression cannot be calibrated independently in each condition, rather the amount 
%     of attention creates the difference of the curves.
%   - If the variable stimulus is an anti-preferred one, additional surround suppression occurs. 
%     However, this effect is weak compared to the feature-based suppression influence, so it is not visible in the data.
%   - Strangely, the data shows also an amplification in condition attend pref. at -120°. 
%     This orientation is different to the preferred orientation, so it still receives strong feature-based suppression
%     in the model and in other data sets, thus no amplification should be observed.
%     Thus with the standard feature-based suppression weights, this effect should not be possible.
%
%\
%> @param setupDir      Directory with experimental setup, legend figures and a mat-file containing all plotting parameters
%> @param debugOn       If true, display additional information for fitting the data
%> @param stdOn         If true, run model with standard parameters
function figure8b(setupDir, debugOn, stdOn)

   if(nargin<1)
      setupDir = 'setups/';
      debugOn = false;
      stdOn = false;
   end
   load(sprintf('%s/plotParams.mat', setupDir));
   addpath('strings/');
   
   
   %% Run Model
   if(~stdOn)
      modelFile = sprintf('%s/modelTreue1999.mat', modelDir);
   else
      modelFile = sprintf('%s/modelTreue1999.mat', modelDirStd);
   end
   
   % Conditions:
   % - 1 => pair with attend away
   % - 2 => pair with attend own
   % - 3 => pair with attend other

   C = 3;              % # Conditions
   N = 13;             % # Subexperiments
   nGlobal = N*C;      % # All experiments
   L = 37;             % # Different stimuli/features
   X = 5;              % # Locations
   Xrf = 5;            % Receptive field size
   x1 = 2;             % Stimulus location 1 in L4
   x2 = 4;             % Stimulus location 2 in L4
   x3 = 2;             % Receptive field center in L2
   l3 = ((L-1)*0.5)+1; % Preferred feature

   %Model parameter (can be changed after creation)
   objModel = initModel(L,X,Xrf);
   if(~stdOn)
      objModel.vSupFeat = 2.4;
      objModel.pSupFeat = 3.5;
      objModel.vFeatL2 = 0.25;
   end

   allIn = cell(nGlobal,1);
   allL4 = cell(nGlobal,1);
   allL2 = cell(nGlobal,1);
   allFEF = cell(nGlobal,1);
   allPFC = cell(nGlobal,1);  
   rModel = zeros(C,N);
   rDistractor = zeros(C,N);

   for s1 = 1:N

      % Stimulus preference at location 1 (l1) and 2(l2)
      % - variable preferred is placed at l1
      % - fixed anti-preferred stimulus is placed at l2
      l1 = (s1-1)/(N-1)*(L-1)+1;
      l2 = L;

      % Generate input population responses
      c = 6/L;
      g = 0.00;
      r1 = symetricalExpCircle(L, l1, g, 1, c);
      r2 = symetricalExpCircle(L, l2, g, 1, c);

      % Generate attention
      c = 6/L;
      g = 0.0;
      a1 = symetricalExpCircle(L, l1, g, 1, c);
      a2 = symetricalExpCircle(L, l2, g, 1, c);   

      for c = 1:C
         s = (s1-1)*C+c; %global experiment index            

         % Set input
         rIn = zeros(L,X);
         rIn(:,x1) = r1;  
         rIn(:,x2) = r2;
         allIn{s} = rIn;


         % Set amplification
         allFEF{s} = zeros(1,X);
         allPFC{s} = zeros(L,1); 
         if(c==2)
            allPFC{s}(:) = a1;
         end
         if(c==3)
            allPFC{s}(:) = a2;
         end            

         %% Run model calculation
         [rL4, rL2] = calculateModel(allIn{s}, allFEF{s}, allPFC{s}, objModel);
         allL4{s} = rL4;
         allL2{s} = rL2;

         rModel(c,s1) = rL2(l3,x3);
         rDistractor(c,s1) = rL2(L,x3);

      end % [conditions]         
   end % [experiments]
   %normalize maximal response to 1 like in the data
   rModel = rModel./max(rModel(:));

   save(modelFile, 'rModel', 'allIn', 'allFEF', 'allPFC', 'allL4', 'allL2', 'C', 'N', 'L', 'nGlobal');
   
   
   %% Plotting
   %% For debug, plot a seleted subexperiment in detail
   if(debugOn)
      figure
      n1 = 1+2*C; % feature: -120°
      subplot(2,1,1)
      plot([ allL2{n1}(:,x3), allL2{n1+1}(:,x3), allL2{n1+2}(:,x3)]);
      legend({'Attend away', 'Attend pref', 'Attend anti-pref'});
      ylim([0,1]);

      subplot(2,1,2)
      plot([allIn{n1}(:,x1), allIn{n1}(:,x2)]);
      legend({'Exc stim1', 'Exc stim2'});
      ylim([0,1]);
   end
   
   %% Create plot for model
   figure
   hold on;
   
   for i=1:3
      plot(rModel(i,:), markersLine{i}, 'linewidth', 3, 'color', colors(i,:), 'markersize', 8, ...
         'LineSmoothing','on');
   end   
   formatPlot(setupDir, colors,  markersLine, markers, noSize, labelSize, true, N);
   
   %% In debug mode, create plot also for distractor
   if(debugOn)
      figure
      hold on;

      for i=1:3
         plot(rDistractor(i,:), markersLine{i}, 'linewidth', 3, 'color', colors(i,:), 'markersize', 8, ...
            'LineSmoothing','on');
      end
      formatPlot(setupDir, colors,  markersLine, markers, noSize, labelSize, true, N);
   end

   %In debug mode, saving the figure is not necessary
   if(debugOn)
      return; 
   end
   
   %% Saving
   fprintf('Saving...\n');
   if(~stdOn)
      name = sprintf('%s/fig8b.png', outputdir);
   else
      name = sprintf('%s/fig8b.png', outputdirStd);
   end
   print(name, '-dpng');
         
end


%% Function to format the plot nicely
function formatPlot(setupDir, colors,  markersLine, markers, noSize, labelSize, isBottom, L)

   ylim([0,1.05]);
   box on   
   
   nX = 7; % # Data points
   ylabel('Response', 'fontsize', labelSize);
   set(gca, 'YTick', [0,1]);
   if(~isBottom)
      xlim([0,L]);
      set(gca, 'XTick', linspace(2,L,nX));
      strXTick = sprintf('%i° ', linspace(-120, 180, nX));
   else
      xlim([1,L]);
      set(gca, 'XTick', linspace(1,L,nX));
      strXTick = sprintf('%i° ', linspace(-180, 180, nX));
   end
   set(gca, 'XTicklabel', strsplit(strXTick));
   set(gca, 'linewidth', 2, 'fontsize', noSize);

   if(isBottom)
      xlabel('Relative orientation', 'fontsize', labelSize);      
      legend({'Att. away', 'Att. pref.', 'Att. anti.'}, 'location', 'southeast');  
   end
   
end
