%% / Experimental setup for sharpening of the neuronal tuning curve
% @author Frederik Beuth
%
% Data: Martinez-Trujillo, Treue (2004), Fig. 4A
%
%% @section X10a1 Significant results (p. 747, right column):
%  - Feature-based attention increases response for the preferred motion (positive modulation) and 
%    decreases it for anti-preferred motion (negative modulation).
%    The modulation ratio changes monotonic from the preferred motion to the anti-preferred one, resulting in a curve with 
%    significant negative slope (Fig. 4B).
%  - Fig. 4A,4B illustrate the slope of a single cell, but the negative slope is also valid for the population data (Fig. 5). 
%  - The point of no modulation is at the half distance between preferred and anti-preferred motion, 
%    i.e. at 90° angular distance of preferred and stimulus motion.
%
%% @section X10a2 Setup:
%  - Stimuli have the same size as RFs (area MT), resulting in sizes from 1° to 12°.
%    => We model the stimulus as large as the RF.
%
%% @section X10a3 Calibration of the fit:
%   - The tuning curve of feature-based attention is similarly wide as the input tuning curve, but does not have a baseline.
%   - Input tuning curve is a broad exponential function with a baseline. Baseline is necessary to simulate the response towards 
%     anti-preferred stimuli.
%   - In the simulation, suppression results from two sources: suppression from anti-preferred features, and 
%     the standard amount of surround suppression.
%   - We use full contrasted stimuli: the original authors specify 55cd/m^2. However, it is difficultly to relate this value to a proportional
%     contrast, so we follow Reynolds and Heeger (2009) which uses full contrasted stimuli.
%
%\
%> @param setupDir      Directory with experimental setup, legend figures and a mat-file containing all plotting parameters
%> @param debugOn       If true, display additional information for fitting the data
%> @param stdOn         If true, run model with standard parameters
function figure8a(setupDir, debugOn, stdOn)

   if(nargin<1)
      setupDir = 'setups/';
      debugOn = false;
      stdOn = false;
   end
   load(sprintf('%s/plotParams.mat', setupDir));
   
    
   %% Run Model
   if(~stdOn)
      modelFile = sprintf('%s/modelMartinez2004.mat', modelDir);
   else
      modelFile = sprintf('%s/modelMartinez2004.mat', modelDirStd);
   end
       
   C = 2;               % # Conditions
   N = 1;               % # Subexperiments
   nGlobal = N*C;       % # All experiments
   orients = -180:30:180; % Different stimuli orientations
   L = numel(orients);  % # Different stimuli/features 
   X = 17;              % # Locations
   Xrf = 5;             % Receptive field size in L2
   x1 = 9;              % Stimulus location in L4
   x3 = 5;              % Receptive field center in L2
   XrfH = (Xrf-1)/2;    % Half of receptive field
   l1 = round((L-1)*0.5)+1; % Preferred orientation 

   %Model parameter (can be changed after creation)
   objModel = initModel(L,X,Xrf);
   if(~stdOn)
      objModel.vSupFeat = 2;  
      objModel.pSupFeat = 3;
   end

   allIn = cell(nGlobal,1);
   allL4 = cell(nGlobal,1);
   allL2 = cell(nGlobal,1);
   allFEF = cell(nGlobal,1);
   allPFC = cell(nGlobal,1);  
   rModel = zeros(C,L);

   for s1 = 1:N

      % Generate input population responses         
      c = 8/L;
      g = 0.3; % base line
      r1 = symetricalExpCircle(L, l1, g, 1, c);

      % Generate feature-base attention
      c = 4/L;
      a1 = symetricalExpCircle(L, l1, 0, 1, c);

      for c = 1:C
         s = (s1-1)*C+c; %global experiment index            

         % Set input
         rIn = zeros(L,X);
         rIn(:,x1-XrfH:x1+XrfH) = repmat(r1,[1,Xrf]);
         allIn{s} = rIn;

         % Set amplifications
         allFEF{s} = zeros(1,X);
         allPFC{s} = zeros(L,1); 
         if(c==2)
            allPFC{s}(:) = a1; 
         end   

         %% Run model calculation
         [rL4, rL2] = calculateModel(allIn{s}, allFEF{s}, allPFC{s}, objModel);
         allL4{s} = rL4;
         allL2{s} = rL2;

         rModel(c,:) = rL2(:,x3);
      end % [conditions]

   end % [experiments]
   %normalize maximal response to 1 like in the data
   maxData = max(rModel(:));
   rModel = rModel./maxData;
   if(debugOn)
      fprintf('Model max=%02f\n', maxData);
   end

   save(modelFile, 'rModel', 'allIn', 'allFEF', 'allPFC', 'allL4', 'allL2', 'C', 'N', 'L', 'nGlobal');
   
   %% Create plot for model  
   figure
   addpath('strings/');   
   hold on;
   
   for i=1:2
      plot(rModel(i,:), markersLine{i}, 'linewidth', 3, 'color', colors(i,:), 'markersize', 8, ...
         'LineSmoothing','on');
   end   
   formatPlot(setupDir, colors,  markersLine, markers, noSize, labelSize, true, L);
   
   %In debug mode, saving the figure and plotting the population response is not necessary
   if(debugOn)
      return; 
   end
   
   %% Saving
   fprintf('Saving...\n');
   if(~stdOn)
      name = sprintf('%s/fig8a.png', outputdir);
   else
      name = sprintf('%s/fig8a.png', outputdirStd);
   end
   print(name, '-dpng');   
   
end


%% Function to format the plot nicely
function formatPlot(setupDir, colors,  markersLine, markers, noSize, labelSize, isBottom, L)

   ylim([0,1.05]);
   xlim([1,L]);
   box on   
   
   nX = 7;
   ylabel('Response', 'fontsize', labelSize);
   
   set(gca, 'YTick', [0,1]);
   set(gca, 'XTick', linspace(1,L,nX));
   strXTick = sprintf('%i° ', linspace(-180, 180, nX));
   set(gca, 'XTicklabel', strsplit(strXTick));
   set(gca, 'linewidth', 2, 'fontsize', noSize);

   if(isBottom)
      xlabel('Relative orientation', 'fontsize', labelSize);      
      legend({'Att. away', 'Att. pref.'}, 'location', 'south');      
   end
   
end
