%% / Experimental setup for scaling of the neuronal tuning curve
% @author Frederik Beuth
%
% Data: McAdams and Maunsell, 1999, Fig. 4
%
%% @section X9a1 Significant results (p. 433, right column):
% - In the population (Fig. 4), attention significantly strengthened responses to the attended stimuli.
% - In individual cells, it was significant in 20% of the cells. For the rest, see Fig. 2A.
% - The population shows no significant changes in tuning width (Fig. 2B) or preferred direction (Fig. 2D), but it shows a 
%   sign. change in the asymptote (Fig. 2C). However, some individual cells deviate from this.
%
%% @section X9a2 Setup:
%  - Stimuli have either sizes between 1°-5° (2xSD of 0.6°-2.2°) or between 0.6°-2.2°. Which version is unclear as 
%    the text is not precisely formulated.
%    They were presented at 1-5° eccentricity. V4 RF size are between 3°-4.8° at this eccentricity (Smith et al., 2001).
%    => We model the stimulus as large as the RF.
%
%% @section X9a3 Calibration of the fit:
%   - Attention factor is equal for all orientations, indicating response gain effect.
%   - The suppression at the peak result from weak surround suppression.
%   - The flank slope indicates an pE value of around 1.5, which demands to decrease vE to 2.
%   - Feature-based suppression occurs from peak of the curve to the borders. The data shows no suppression,
%     thus cancels out the suppression with a broad tuning of the input and a baseline
%
%\
%> @param setupDir      Directory with experimental setup, legend figures and a mat-file containing all plotting parameters
%> @param debugOn       If true, display additional information for fitting the data
%> @param stdOn         If true, run model with standard parameters
function figure7(setupDir, debugOn, stdOn)

   if(nargin<1)
      setupDir = 'setups/';
      debugOn = false;
      stdOn = false;
   end
   load(sprintf('%s/plotParams.mat', setupDir));
   
   %% Run Model
   if(stdOn)
      modelFile = sprintf('%s/modelMcAdams1999.mat', modelDirStd);
   else      
      modelFile = sprintf('%s/modelMcAdams1999.mat', modelDir);
   end
   C = 2;               % # Conditions
   N = 1;               % # Subexperiments
   nGlobal = N*C;       % # All experiments
   orients = -180:30:180; % Different stimuli orientations
   L = numel(orients);  % # Different stimuli/features 
   X = 17;              % # Locations
   Xrf = 5;             % receptive field size in L2
   x1 = 9;              % Stimulus location in L4
   x3 = 5;              % Receptive field center in L2
   XrfH = (Xrf-1)/2;    % Half of receptive field

   % Model parameter (can be changed after creation)
   objModel = initModel(L,X,Xrf);
   if(~stdOn)
      objModel.vE = 2;
      objModel.pE = 1.5;
      objModel.vSupSur = 0.75;
      objModel.baseline = 0.1;
   end

   allIn = cell(nGlobal,1);
   allL4 = cell(nGlobal,1);
   allL2 = cell(nGlobal,1);
   allFEF = cell(nGlobal,1);
   allPFC = cell(nGlobal,1);  
   rModel = zeros(C,L);

   for s1 = 1:N

      l1 = (L-1)/2+1; % preferred feature of the cell

      % Generate input population responses
      c = 8/L;
      g = 0.1; % Baseline
      cMax = 0.5; % maximal contrast
      r1 = symetricalExpCircle(L, l1, g, 1, c) * cMax;

      for c = 1:C
         s = (s1-1)*C+c; %global experiment index            

         % Set input
         rIn = zeros(L,X);
         rIn(:,x1-XrfH:x1+XrfH) = repmat(r1,[1,Xrf]);
         allIn{s} = rIn;

         % Set amplification
         allFEF{s} = zeros(1,X);
         allPFC{s} = zeros(L,1); 
         if(c==2)
            allFEF{s}(x1-XrfH:x1+XrfH) = 1; 
         end   

         %% Run model calculation
         [rL4, rL2] = calculateModel(allIn{s}, allFEF{s}, allPFC{s}, objModel);
         allL4{s} = rL4;
         allL2{s} = rL2;

         rModel(c,:) = rL2(:,x3);
      end % [conditions]

   end % [experiments]
   %normalize maximal response to 1 like in the data
   maxData = max(rModel(:));
   rModel = rModel./maxData;
   if(debugOn)
      fprintf('Maximal model reponse=%0.3f\n', maxData);
   end

   save(modelFile, 'rModel', 'allIn', 'allFEF', 'allPFC', 'allL4', 'allL2', 'C', 'N', 'L', 'nGlobal');
   
   %% Create plot for model
   figure
   hold on;
   
   for i=1:2
      plot(rModel(i,:), markersLine{i}, 'linewidth', 3, 'color', colors(i,:), 'markersize', 8, ...
         'LineSmoothing','on');
   end   
   formatPlot(setupDir, colors,  markersLine, markers, noSize, labelSize, true, L);
   
   %In debug mode, saving the figure and plotting the population response is not necessary
   if(debugOn)
      return; 
   end

   %% Saving
   fprintf('Saving...\n');
   figId = 7;
   if(stdOn)
      name = sprintf('%s/fig%i.png', outputdirStd, figId);
   else
      name = sprintf('%s/fig%i.png', outputdir, figId);
   end
   print(name, '-dpng');
         
end


%% Function to format the plot nicely
function formatPlot(setupDir, colors,  markersLine, markers, noSize, labelSize, isBottom, L)

   ylim([0,1.05]);
   xlim([1,L]);
   box on   
   
   nX = 7;
   ylabel('Response', 'fontsize', labelSize);
   set(gca, 'YTick', [0,1]);
   set(gca, 'XTick', linspace(1,L,nX));
   strXTick = sprintf('%i° ', linspace(-90, 90, nX));
   set(gca, 'XTicklabel', strsplit(strXTick));
   set(gca, 'linewidth', 2, 'fontsize', noSize);

   if(isBottom)
      xlabel('Relative orientation', 'fontsize', labelSize);      
      legend({'Att. away', 'Att. pref.'}, 'location', 'southeast');  
   end
   
end
