%% / Experimental setup for mix between response and contrast gain
% @author Frederik Beuth
%
% Data: Williford and Maunsell, 2006, Fig. 6G
%
%% @section X12a1 Significant results (p. 45, right column):
% - To distinguish between response and contrast gain, they investigate only on neurons which significantly saturated at lower contrasts and
%   also shown a significant attentional modulation (Fig. 6G). Both together describes the response gain effect.
% - They found at least a partial response gain as attention increases also the saturated response at high contrasts.
%
%% @section X12a2 Setup:
%  - Stimuli sizes and RF sizes are not reported in the publication. From the figures, they seem to be equal. 
%    Reynolds & Heeger, 2009 model this experiment with stimuli as large as the RFs.
%    => We model the stimulus as large as the RF too.
%
%% @section X12a3 Calibration of the fit:
% - The data shows neurons which saturate at lower contrast, thus we choose higher values for the contrast response 
%   parameters (sigma, vE, pE).
% - Calibrate amount of attention in parallel with amount of suppression.
%
%\
%> @param setupDir      Directory with experimental setup, legend figures and a mat-file containing all plotting parameters
%> @param debugOn       If true, display additional information for fitting the data
%> @param stdOn         If true, run model with standard parameters
function figure6(setupDir, debugOn, stdOn)

   if(nargin<1)
      setupDir = 'setups/';
      debugOn = false;
      stdOn = false;
   end
   load(sprintf('%s/plotParams.mat', setupDir));
   
   %% Run model 
   % Contrasts: The method sections of Williford2006 states that they use 11 values with 0% and the 
   % rest logarithmically spaced between 0.02 and 1: [0, logspace(-1.7,0,10)]. However, Fig. 6G shows 
   % rounded values, so we also use rounded values:
   contrastsData = [0,0.02, 0.03, 0.04, 0.06, 0.10, 0.16, 0.25, 0.4, 0.65, 1];
   
   
   %if simulation data already exists, skip model run and use the old results
   if(~stdOn)
      modelFile = sprintf('%s/modelWilliford2006.mat', modelDir);
   else
      modelFile = sprintf('%s/modelWilliford2006.mat', modelDirStd);
   end
   % Conditions:
   % - 1 => stimulus with attend away
   % - 2 => stimulus with attend own      
   C = 2;               % # Conditions
   N = 11;              % # Subexperiments/contrasts
   contrasts = contrastsData;
   nGlobal = N*C;       % # All experiments
   L = 9;               % # Different stimuli/features 
   X = 17;              % # Locations
   Xrf = 5;             % receptive field size in L2
   x1 = 9;              % Stimulus location in L4
   x3 = 5;              % Receptive field center in L2
   XrfH = (Xrf-1)/2;    % Half of receptive field

   %preferred orientation
   l1 = round((L-1)*0.5)+1;

   %Model parameter (can be changed after creation)
   objModel = initModel(L,X,Xrf);
   if(~stdOn)
      objModel.vSupSur = 1;
   end

   allIn = cell(nGlobal,1);
   allL4 = cell(nGlobal,1);
   allL2 = cell(nGlobal,1);
   allFEF = cell(nGlobal,1);
   allPFC = cell(nGlobal,1);
   rModel = zeros(2,N);

   for ic1 = N:-1:1

      % Create input: black background with a grating in the middle
      c = 8/L;
      g = 0.0;
      r1 = symetricalExpCircle(L, l1, g, 1, c);

      for c = 1:C
           s = (ic1-1)*C+c; %global experiment index       

           rIn = zeros(L,X);
           rIn(:,x1-XrfH:x1+XrfH) = repmat(contrasts(ic1) * r1,[1,Xrf]);
           allIn{s} = rIn;

           allFEF{s} = zeros(1,X);
           allPFC{s} = zeros(L,1);
           if(c==2)
              allFEF{s}(x1-XrfH:x1+XrfH) = 1;
           end

           %% Run model calculation
           [rL4, rL2] = calculateModel(allIn{s}, allFEF{s}, allPFC{s}, objModel);
           allL4{s} = rL4;
           allL2{s} = rL2;

           rModel(c, ic1) = rL2(l1,x3); 
       end
   end
   save(modelFile, 'rModel', 'allIn', 'allFEF', 'allPFC', 'allL4', 'allL2', 'C', 'N', 'L', 'nGlobal', 'contrasts', 'objModel', ...
      'x1','x3', 'l1');
   
   % Normalize model response as data responses
   maxData = max(rModel(:));
   minData = min(rModel(:));
   if(debugOn)
      fprintf('Model max=%02f, min = %02f\n', maxData, minData);
   end
    rModel = (rModel - minData) ./ (maxData - minData);
   
   %% Create plot for model
   figure
   
   for i=1:2
      semilogx(contrasts, rModel(i,:), markersLine{i}, 'linewidth', 3, 'color', colors(i,:), 'markersize', 8, ...
         'LineSmoothing','on');
      hold on;
   end   
   formatPlot(setupDir, colors,  markersLine, markers, noSize, labelSize, true, N, contrasts);
  
   %In debug mode, saving the figure and plotting the population response is not necessary
   if(debugOn)
      return; 
   end
   
   %% Saving
   fprintf('Saving...\n');
   if(~stdOn)
      name = sprintf('%s/fig6.png', outputdir);
   else
      name = sprintf('%s/fig6.png', outputdirStd);
   end
   print(name, '-dpng');

end


%% Function to format the plot nicely
function formatPlot(setupDir, colors,  markersLine, markers, noSize, labelSize, isBottom, N, contrasts)

   ylim([0,1.05]);
   xlim([min(contrasts),max(contrasts)]);
   box on   
   
   ylabel('Response', 'fontsize', labelSize);
   set(gca, 'YTick', [0,1]);
   set(gca, 'linewidth', 2, 'fontsize', noSize);

   if(isBottom)
      xlabel('Log contrast', 'fontsize', labelSize);      
      legend({'Att. away', 'Att. pref.'}, 'location', 'southeast');
   end
   
end
