%% / Experimental setup for contrasts gain
% @author Frederik Beuth
%
% Data: Reynolds, Pasternak, Desimone 2000, Fig. 5A
%
%% @section X11a1 Significant results (p. 706, left column):
%  - Attention increases responses significantly. Larger at lower contrasts, quantified in Fig. 6.
%  - Attention increases even the responses below the neurons contrast-response thresholds significantly. Thus attention reduced 
%    contrast–response thresholds.
%
%% @section X11a2 Setup:
%  - Stimuli sizes are between 1.5-2°, eccentricity is unknown. However, this size is very small compared to V4 rf sizes at all eccentricities, 
%    so we assume that a stimulus falls only in the receptive field of a single V4.
%
%% @section X11a3 Calibration of the fit:
% - Calibrate contrast response parameters (sigma, vE, pE) and amount of attention in parallel.
% - The chosen values for the contrast response parameters are used as standard values.
%
%\
%> @param setupDir      Directory with experimental setup, legend figures and a mat-file containing all plotting parameters
%> @param debugOn       If true, display additional information for fitting the data
%> @param stdOn         If true, run model with standard parameters
function figure5(setupDir, debugOn, stdOn)

   if(nargin<1)
      setupDir = 'setups/';
      debugOn = false;
      stdOn = false;
   end
   load(sprintf('%s/plotParams.mat', setupDir));

   %% Run Model
   % The method sections of Reynolds2000 states that they use seven values : [0.0125, 0.025, 0.05, 0.1, 0.2, 0.4, 0.8]
   % However, the data in the figure shows 0% plus five contrasts for which the neuron reacts. I guess from Fig. 3C: [0.05, 0.1, 0.2, 0.4, 0.8]
   % For plotting 0% with semilogx(), we use 0.025
   contrastsData = [0.025, 0.05, 0.1, 0.2, 0.4, 0.8]; 
   
   if(~stdOn)
      modelFile = sprintf('%s/modelReynolds2000.mat', modelDir);
   else
      modelFile = sprintf('%s/modelReynolds2000.mat', modelDirStd);
   end
   
   % Conditions:
   % - 1 => stimulus with attend away
   % - 2 => stimulus with attend own     
   C = 2;               % # Conditions
   N = 6;               % # Subexperiments
   contrasts = [0.0, 0.05, 0.1, 0.2, 0.4, 0.8]; % # Contrasts like in the experiment (see above)
   nGlobal = N*C;       % # All experiments
   L = 9;               % # Different stimuli/features 
   X = 17;              % # Locations
   Xrf = 5;             % Receptive field size
   x1 = 9;              % Stimulus location in L4
   x3 = 5;              % Receptive field center in L2
   XrfH = (Xrf-1)/2;    % Half of receptive field 

   %preferred orientation
   l1 = round((L-1)*0.5)+1;

   %Model parameter (can be changed after creation)
   objModel = initModel(L,X,Xrf);
   if(~stdOn)
      objModel.vE = 2;  
      objModel.pE = 1.75;  
   end

   allIn = cell(nGlobal,1);
   allL4 = cell(nGlobal,1);
   allL2 = cell(nGlobal,1);
   allFEF = cell(nGlobal,1);
   allPFC = cell(nGlobal,1);
   rModel = zeros(C,N);

   for ic1 = 1:N

      % Create input: black background with a grating in the middle
      c = 8/L;
      g = 0.0;
      r1 = symetricalExpCircle(L, l1, g, 1, c);

      for c = 1:C
           s = (ic1-1)*C+c; %global experiment index       

           rIn = zeros(L,X);
           rIn(:,x1) = contrasts(ic1) * r1;
           allIn{s} = rIn;

           allFEF{s} = zeros(1,X);
           allPFC{s} = zeros(L,1);
           if(c==2)
              allFEF{s}(x1-XrfH:x1+XrfH) = 1;
           end

           %% Run model calculation
           [rL4, rL2] = calculateModel(allIn{s}, allFEF{s}, allPFC{s}, objModel);
           allL4{s} = rL4;
           allL2{s} = rL2;

           rModel(c, ic1) = rL2(l1,x3); 
       end
   end
   save(modelFile, 'rModel', 'allIn', 'allFEF', 'allPFC', 'allL4', 'allL2', 'C', 'N', 'L', 'nGlobal', 'contrasts', 'l1', 'x1', 'x3');
   
   % normalize model response as the data
   maxData = max(rModel(:));
   minData = min(rModel(:));
   if(debugOn)
      fprintf('Model max=%02f, min = %02f\n', maxData, minData);
   end
   rModel = (rModel - minData) ./ (maxData - minData);
   
   %% Create plot for model
   figure
   
   for i=1:2
      semilogx(contrastsData, rModel(i,:), markersLine{i}, 'linewidth', 3, 'color', colors(i,:), 'markersize', 8, ...
         'LineSmoothing','on');
      hold on;
   end   
   formatPlot(setupDir, colors,  markersLine, markers, noSize, labelSize, true, N, contrasts);
   
   %In debug mode, saving the figure and plotting the population response is not necessary
   if(debugOn)
      return; 
   end
   
   %% Saving
   fprintf('Saving...\n');
   if(~stdOn)
      name = sprintf('%s/fig5.png', outputdir);
   else
      name = sprintf('%s/fig5.png', outputdirStd);
   end
   print(name, '-dpng');
   
end


%% Function to format the plot nicely
function formatPlot(setupDir, colors,  markersLine, markers, noSize, labelSize, isBottom, N, contrasts)

   ylim([0,1.05]);
   xlim([min(contrasts),max(contrasts)]);
   box on   
   
   ylabel('Response', 'fontsize', labelSize);
   set(gca, 'YTick', [0,1]);
   set(gca, 'linewidth', 2, 'fontsize', noSize);

   if(isBottom)
      xlabel('Log contrast', 'fontsize', labelSize);      
      legend({'Att. away', 'Att. pref.'}, 'location', 'southeast');      
   end
   
end
