%% / Experimental setup for biased competition with feature-based attention 
% @author Frederik Beuth
%
% Data: Chelazzi et al., 1998, Fig. 10A and 10B
%
%% @section X13a1 Significant results (p. 2927, right column):
% - The results of Fig. 10A/B were not tested for significance. However, the authors reported that
%   the anti-preferred ('poor') stimulus has a suppressive effect on the response to the preferred ('good') stimulus.
%
%% @section X13a2 Setup:
%  - Receptive fields in IT are very large, so we assume that a stimulus falls only in a single rf.
%
%% @section X13a3 Calibration of the fit:
%   - 1) Choose tuning curve and baseline
%   - 2) Choose amplification
%   - 3) Choose feature-based suppression
%
%\
%> @param setupDir      Directory with experimental setup, legend figures and a mat-file containing all plotting parameters
%> @param debugOn       If true, display additional information for fitting the data
%> @param stdOn         If true, run model with standard parameters
function figure4(setupDir, debugOn, stdOn)

   if(nargin<1)
      setupDir = 'setups/';
      debugOn = false;
      stdOn = false;
   end
   load(sprintf('%s/plotParams.mat', setupDir));
   addpath('strings/');   

   
   %% Run Model
   if(~stdOn)
      modelFile = sprintf('%s/modelChelazzi1998.mat', modelDir);
   else
      modelFile = sprintf('%s/modelChelazzi1998.mat', modelDirStd);
   end
      
   % Conditions:
   % - 1 => preferred stimulus alone (attend away)
   % - 2 => anti-preferred stimulus alone (attend away)
   % - 3 => pair with attend away
   % - 4 => pair with attend own
   % - 5 => pair with attend other      
   C = 5;             % # Conditions
   N = 1;             % # Subexperiments
   nGlobal = N*C;     % # All experiments
   L = 33;            % # Different stimuli/features
   X = 9;             % # Locations
   Xrf = 5;           % # Receptive field size      
   x1 = 4;            % Stimulus location 1 in L4
   x2 = 6;            % Stimulus location 2 in L4
   x3 = 3;            % Receptive field center in L2
   % Stimulus size is fixed to 1

   %Model parameter (can be changed after creation)
   objModel = initModel(L,X,Xrf);
   if(~stdOn)    
      objModel.pSupFeat = 3.0;   
      objModel.vFeatL2 = 0.17;      
      objModel.baseline = 0.15;
   end

   allIn = cell(nGlobal,1);
   allL4 = cell(nGlobal,1);
   allL2 = cell(nGlobal,1);
   allFEF = cell(nGlobal,1);
   allPFC = cell(nGlobal,1);  
   rModel = zeros(C,N);

   for s1 = 1:N

      % Stimulus preference at location 1 (l1) and 2(l2)
      % - preferred (l1) and anti-preferred stimulus (l2) 
      l1 = ((L-1)*0.25)+1;
      l2 = ((L-1)*0.75)+1;

      % Generate input response
      c = 8/L;
      g = 0.1; %baseline
      r1 = symetricalExpCircle(L, l1, g, 1, c); 
      r2 = symetricalExpCircle(L, l2, g, 1, c);

      % Generate attention signal
      a1 = symetricalExpCircle(L, l1, 0, 1, c);
      a2 = symetricalExpCircle(L, l2, 0, 1, c);

      for c = 1:C
         s = (s1-1)*C+c; %global experiment index            

         % Set input
         if(c==1)
            rIn = zeros(L,X);
            rIn(:,x1) = r1; 
         elseif(c==2)
            rIn = zeros(L,X);
            rIn(:,x2) = r2;
         else
            rIn = zeros(L,X);
            rIn(:,x1) = r1;  
            rIn(:,x2) = r2;
         end            
         allIn{s} = rIn;

         % Set amplification
         allFEF{s} = zeros(1,X);
         allPFC{s} = zeros(L,1); 
         if(c==4)
            allPFC{s}(:) = a1;
         end
         if(c==5)
            allPFC{s}(:) = a2;
         end            

         %% Run model calculation
         [rL4, rL2] = calculateModel(allIn{s}, allFEF{s}, allPFC{s}, objModel);
         allL4{s} = rL4;
         allL2{s} = rL2;

         rModel(c,s1) = rL2(l1,x3);
      end % [conditions]

   end % [experiments]

   % Debugging: plot tuning curves
   if(debugOn)

      aa = 3; %<-pair with attend away
      %aa = 4; %<-pair with attend pref.
      figure
      subplot(2,2,1)
      plot([allIn{1}(:,x1), allIn{2}(:,x2), allIn{aa}(:,x1), allIn{aa}(:,x2)])
      title('Input');

      subplot(2,2,2)
      plot([allL4{1}(:,x1), allL4{2}(:,x2), allL4{aa}(:,x1), allL4{aa}(:,x2)])
      title('Layer 4');

      subplot(2,2,3)
      plot([allL2{1}(:,x3), allL2{2}(:,x3), allL2{aa}(:,x3)])
      title('Layer 2/3');
   end

   %normalize maximal response to 1 like in the data
   rModel = rModel./max(rModel(:));

   save(modelFile, 'rModel', 'allIn', 'allFEF', 'allPFC', 'allL4', 'allL2', 'C', 'N', 'L', 'nGlobal');
      
   %% Create plot for model
   figure
   bar(rModel, 0.3, 'FaceColor', colors(6,:), 'EdgeColor', 'none');   
   formatPlot(setupDir, colors,  markersLine, markers, noSize, labelSize, true, size(rModel,1));   
   
   %In debug mode, saving the figure and plotting the population response is not necessary
   if(debugOn)
      return; 
   end
   
   %% Saving
   fprintf('Saving...\n');
   figId = 4;
   if(~stdOn)
      name = sprintf('%s/fig%i.png', outputdir, figId);
   else
      name = sprintf('%s/fig%i.png', outputdirStd, figId);
   end
   print(name, '-dpng');
   
end


%% Function to format the plot nicely
function formatPlot(setupDir, colors,  markersLine, markers, noSize, labelSize, isBottom, L)

   ylim([0,1.05]);
   xlim([0.3,L+0.7]);
   box off
   
   nX = 5; % # Bars
   ylabel('Response', 'fontsize', labelSize);
   set(gca, 'YTick', [0,1]);
   set(gca, 'XTick', linspace(1,L,nX));
   if(isBottom)
      set(gca, 'XTicklabel', 1:5);
   else
      set(gca, 'XTicklabel', []);
   end
   set(gca, 'linewidth', 2, 'fontsize', noSize, 'Layer', 'top');
   
end
