%% / Experimental setup for contrast dependent competition
% @author Frederik Beuth
%
% Data: Busse, Wade, Carandini (2009), Fig. 4A
%
%% @section X3c1 Significant results (p. 932 left)
% - Significance tests were not conducted, but the authors reported the following results:
% - Single stimulus: contrast changes modulate the amplitude of a tuning curve, but not its width. This finding is valid for single neurons as well as for the
%   population response.
% - Overlaid stimuli (plaid): 
%   - Similar contrast (6% and 12%, or 12% and 12%) result in two peaks in the population response
%   - Large different contrast result in one peak in the population response
%   - Response to pair is lower as responses to single stimuli (cross-orientation suppression)
%
%% @section X3c2 Setup
% - Vary the stimulus contrast over the experiments with values of 0%, 12%, or 50% contrast. Use only these 3 values to prevent cluttering of the plot
% - Record the full neuronal population in V1 (used 13 neurons in the model like in the data)
% - Stimuli are oriented edges of 0° and 90°, hence use neurons 4 and 10 
% - Experiment uses overlapping stimuli, so place both stimuli at the same place
% - Experiment uses no attention, so record the neurons only in the attend away conditions
% - Stimuli sizes and RF sizes were not reported. We could not investigate the amount of surround suppression as it would be visible 
%   only in an attention condition which is not part of the data. Thus we assume no suppression as it is the simplest case, hence we 
%   assume that a stimulus falls only in a single receptive field.
%
%% @section X3c3 Calibration of the fit:
% - 1) Calibrate contrast response function parameters (sigma, eV, eP) to fit the peaks in conditions with a single stimulus
% - 2) Set width and baseline of input tuning function
% - 3) Calibrate amount of inhibition to fit data of conditions with a stimulus pair 
%
%\
%> @param setupDir      Directory with experimental setup, legend figures and a mat-file containing all plotting parameters
%> @param debugOn       If true, display additional information for fitting the data
%> @param stdOn         If true, run model with standard parameters
function figure3f(setupDir, debugOn, stdOn)

   if(nargin<1)
      setupDir = 'setups/';
      debugOn = false;
      stdOn = false;
   end  
   load(sprintf('%s/plotParams.mat', setupDir));
   sizeFig = [0.5, 0.5, 6, 15]; 
   
   c1s = [0 12 50]; % Contrasts: available are [0 6 12 25 50 100]
   c2s = c1s;
   nc1s = length(c1s);
   nc2s = length(c2s);  

   
   %% Run model  
   L = 13;         % # Orientations per experiment
   X = 9;          % # Coordinates in space
   Xrf = 5;        % # RF size
   if(~stdOn)
      modelFile = sprintf('%s/modelBusse2009.mat', modelDir);
   else
      modelFile = sprintf('%s/modelBusse2009.mat', modelDirStd);
   end      
      
   %preferred orientation
   l1 = round((L-1)*0.25)+1;
   l2 = round((L-1)*0.75)+1;

   nGlobal = nc1s * nc2s;

   %Model parameter (can be changed after creation)
   objModel = initModel(L,X,Xrf);
   if(~stdOn)  
      objModel.vSupFeat = 2.5;  
      objModel.vE = 3.5;        
      objModel.baseline = 0.05;
   end

   allIn = cell(nGlobal,1);
   allL4 = cell(nGlobal,1);
   allL2 = cell(nGlobal,1);   
   rModel = zeros(L,nc1s,nc2s);

   s = 1;
   for ic1 = 1:nc1s
       for ic2 = 1:nc2s

           % Create input: black background with an overlapping grating in the middle
           c = 8/L;
           g = 0.15;
           r1 = symetricalExpCircle(L, l1, g, 1, c); 
           r2 = symetricalExpCircle(L, l2, g, 1, c);

           rIn = zeros(L,X);
           rIn(:,5) = max(c1s(ic1)/100 * r1, c2s(ic2)/100 * r2);
           allIn{s} = rIn;

           rFEF = zeros(1,X);
           rPFC = zeros(L,1); 

           %% Run model calculation
           [rL4, rL2] = calculateModel(rIn, rFEF, rPFC, objModel);
           allL4{s} = rL4;
           allL2{s} = rL2;

           rModel(:, ic2, ic1) = rL2(:,3);       
           s = s+1;
       end
   end
   save(modelFile, 'rModel', 'allIn', 'rFEF', 'rPFC', 'allL4', 'allL2', 'nc1s', 'nc2s', 'L', 'nGlobal');
  
   
   %% Create plot for model   
   figure   
   pos2 = get(gca, 'position');
   hAxisMain = gca;
   hold on;
   
   for i=1:3
      for j=1:3
         pos3 = [pos2(1) + (0.31*(i-1)+0.07)*pos2(3), pos2(2) + (1-0.31*j+0.01)*pos2(4), 0.23*pos2(3), 0.23*pos2(4)];
         axes('position',pos3, 'nextplot', 'add');               
         plot(rModel(:,i,j), 'o', 'linewidth', 2, 'color', colors(1,:), 'markersize', 6, 'MarkerFaceColor', colors(1,:));
         plot(rModel(:,i,j), '-', 'linewidth', 3, 'color', colors(1,:), 'LineSmoothing','on');
         formatPlot(i,j,L, noSize);               
      end
   end
   
   set(gcf, 'currentaxes', hAxisMain); 
   box on;
   axis([0,1,0,1.05]);         
   
   set(gca, 'YTick', []);
   set(gca, 'XTick', []);         
   set(gca, 'linewidth', 2, 'fontsize', noSize);
   
   %In debug mode, saving the figure and plotting the population response is not necessary
   if(debugOn)
      return; 
   end
   
   %% Saving
   fprintf('Saving...\n');
   if(~stdOn)
      name = sprintf('%s/fig3f.png', outputdir);
   else
      name = sprintf('%s/fig3f.png', outputdirStd);
   end
   print(name, '-dpng');
end


%% Function to format the plot nicely
function formatPlot(i, j, L, noSize)

   ylim([-0.05, 1.05]);
   xlim([1,L]);
   axis off;

   if(~(i==3 && j==3))
      
   end
   if(i==1) 

      if(j==1)
         text(-7,0.5,'0%','color', 'k', 'Rotation', 90, 'HorizontalAlignment', 'center', 'fontsize', noSize);
      end
      if(j==2)
         text(-7,0.5,'12%','color', 'k', 'Rotation', 90, 'HorizontalAlignment', 'center', 'fontsize', noSize);
      end
      if(j==3)
         text(-7,0.5,'50%','color', 'k', 'Rotation', 90, 'HorizontalAlignment', 'center', 'fontsize', noSize);
      end
   end

   if(j==3)
      
      if(i==1)
         text(L/2,-0.65,'0%','color', 'k', 'HorizontalAlignment', 'center', 'fontsize', noSize);
      end
      if(i==2)
         text(L/2,-0.65,'12%','color', 'k', 'HorizontalAlignment', 'center', 'fontsize', noSize);
      end
      if(i==3)
         text(L/2,-0.65,'50%','color', 'k', 'HorizontalAlignment', 'center', 'fontsize', noSize);
      end
      
   end

   if(i==3 && j==3)
      line([1,L], [0,0], 'linewidth', 2, 'color', 'k');
      xPos = [floor(L*0.25)+1, floor(L*0.75)+1]; 
      line([xPos(1),xPos(1)], [0,-0.05], 'linewidth', 2, 'color', 'k');
      line([xPos(2),xPos(2)], [0,-0.05], 'linewidth', 2, 'color', 'k');
      text(xPos(1), -0.22, '0\circ', 'HorizontalAlignment', 'center', 'fontsize', round(noSize*0.8), 'interpreter', 'tex');
      %+0.1 by 90° to shift label to the right a little bit
      text(xPos(2)+L*0.1, -0.22, '90\circ', 'HorizontalAlignment', 'center', 'fontsize', round(noSize*0.8), 'interpreter', 'tex');
   else
      set(gca, 'XTick', []);
      set(gca, 'YTick', []);
   end
end
