%% / Experimental setup for biased competition - selectivity in relation to sensory interaction
% @author Frederik Beuth
%
% Data: Reynolds, Chelazzi, Desimone (1999), Fig. 11a,b,d
%
%% @section X3c1 Significant results (p. 1746 right bottom)
% - 68% of the neurons are modulated by attention. The following results concern this sub-population:
% - Attend away:
%   - Slope of the fitted line is 0.49, which is not significantly different from 0.5
%   - The line is significantly shifted upward by 0.06, indicating an increased response due to adding the second stimuli
% - Attend probe (mostly anti-preferred) stimulus:
%   - Slope is significantly increased from 0.49 to 0.83, which is significantly different from 0.5
% - Attend reference (mostly preferred) stimulus:
%   - Slope is significantly decreased from 0.49 to 0.21, which is significantly different from 0.5
%
%% @section X3b2 Setup
% - Record an arbitrary neuron in V4
% - Use a pool of 16 stimuli with a mix of different orientations and colors
% - In each experiment, arbitrary select a reference and a probe stimulus from them. They are arbitrary selected to investigate all possible stimulus
%   conditions
% - Record neuron in all of the 9 conditions in each experiment
% - Calculate for each experiment the selectivity and sensory interaction for all 3 attention conditions
% - Stimuli have a width of 0.25° and a heigth of 1°, eccentricity is unknown. However, this size is very small compared to V4 rf sizes at all eccentricities, 
%   so we assume that a stimulus falls only in the receptive field of a single V4 location.
%
%% @section X3b3 Calibration of the fit:
% - Pretty standard biased competition
% - The unattended condition will mostly result in the correct slope of ~0.5
% - Calibrate first amplification to fit attend reference condition
% - Afterwards, calibrate suppression to fit attend probe case
%
%\
%> @param setupDir      Directory with experimental setup, legend figures and a mat-file containing all plotting parameters
%> @param debugOn       If true, display additional information for fitting the data
%> @param stdOn         If true, run model with standard parameters
function figure3cde(setupDir, debugOn, stdOn)

   if(nargin<1)
      setupDir = 'setups/';
      debugOn = false;
      stdOn = true;
   end
   load(sprintf('%s/plotParams.mat', setupDir));
   plotPositions = [0.1, 0.16, 0.22, 0.37*2;
                    0.7, 0.16, 0.22, 0.37*2;
                    0.4, 0.16, 0.22, 0.37*2];

   
   %% Run model
   %conditions:
   % 1 => reference alone at location 1, attend away
   % 2 => reference alone at location 1, attend reference: spatial attend to location 1 
   % 3 => reference alone at location 1, attend probe: spatial attend to location 3
   % 4 => probe alone at location 3, attend away
   % 5 => probe alone at location 3, attend reference
   % 6 => probe alone at location 3, attend probe
   % 7 => pair attend away
   % 8 => pair attend reference
   % 9 => pair attend probe

   if(~stdOn)
      modelFile = sprintf('%s/modelReynolds1999.mat', modelDir);
   else
      modelFile = sprintf('%s/modelReynolds1999.mat', modelDirStd);
   end
   C = 9;             % # Conditions
   N = 200;           % # Subexperiments/sessions
   nGlobal = N*C;     % # All experiments
   L = 16;            % # Different stimuli/features
   X = 9;             % # Locations
   Xrf = 5;           % Receptive field size      
   x1 = 4;            % Stimulus location 1
   x2 = 6;            % Stimulus location 2
   x3 = 3;            % Receptive field center

   %Model parameter (can be changed after creation)
   objModel = initModel(L,X,Xrf);
   if(~stdOn)      
      objModel.vSP = 1.25;
      objModel.vSupFeat = 3.5;
      objModel.pSupFeat = 3;
   end

   allIn = cell(nGlobal,1);
   allL4 = cell(nGlobal,1);
   allL2 = cell(nGlobal,1);
   allFEF = cell(nGlobal,1);
   allPFC = cell(nGlobal,1);      
   ssiModel = zeros(N,2,3);
   for s1 = 1:N

      %choose the orientation/color of reference (l1) and probe stimulus (l2) arbitrarily
      l1 = randi(L); l2 = randi(L);
      %generate input
      c = 8/L;
      g = 0.1;
      r1 = symetricalExpCircle(L, l1, g, 1, c);
      r2 = symetricalExpCircle(L, l2, g, 1, c);

      for c = 1:C
         s = (s1-1)*C+c; %global experiment index 

         % Set input
         %reference alone
         if(c<=3)
            rIn = zeros(L,X);
            rIn(:,x1) = r1;
            allIn{s} = rIn;

         %probe alone
         elseif(c<=6)
            rIn = zeros(L,X);
            rIn(:,x2) = r2; 
            allIn{s} = rIn;
         %pair
         elseif(c<=9)
            rIn = zeros(L,X);
            rIn(:,x1) = r1;
            rIn(:,x2) = r2;
            allIn{s} = rIn;
         else
            allIn{s} = allIn{s-1};
         end            

         % Set amplification
         allFEF{s} = zeros(1,X);
         allPFC{s} = zeros(L,1); 
         if(c==2 || c==5 || c==8)
            allFEF{s}(x1) = 1;
         end
         if(c==3 || c==6 || c==9)
            allFEF{s}(x2) = 1;
         end            

         %% Run model calculation
         [rL4, rL2] = calculateModel(allIn{s}, allFEF{s}, allPFC{s}, objModel);
         allL4{s} = rL4;
         allL2{s} = rL2;

         if(mod(s, 20) == 1)
            fprintf('.');
         end
      end

      %% Calculate selectivity and sensory interaction for the model
      s = (s1-1)*C; %global experiment index of condition 0
      % reference => l1, probe => l2
      % 1 => selectivity (probe - ref)
      % 2 => sensory interaction (pair - ref)
      l3 = round(L/2); % recorded neuron
      ssiModel(s1,1,1) = allL2{s+4}(l3,x3) - allL2{s+1}(l3,x3); % condition 'AttendAway'
      ssiModel(s1,2,1) = allL2{s+7}(l3,x3) - allL2{s+1}(l3,x3);
      ssiModel(s1,1,2) = allL2{s+5}(l3,x3) - allL2{s+2}(l3,x3); % condition 'AttendRef'
      ssiModel(s1,2,2) = allL2{s+8}(l3,x3) - allL2{s+2}(l3,x3);
      ssiModel(s1,1,3) = allL2{s+6}(l3,x3) - allL2{s+3}(l3,x3); % condition 'AttendProbe'
      ssiModel(s1,2,3) = allL2{s+9}(l3,x3) - allL2{s+3}(l3,x3);         

   end %[experiments]
   save(modelFile, 'ssiModel', 'allIn', 'allFEF', 'allPFC', 'allL4', 'allL2', 'C', 'N', 'L', 'nGlobal');
   fprintf('\n');
   

   %% Create plot for model
   figure
   for i=1:3
      subplot('position', plotPositions(i,:));
      hold on;
      plot(squeeze(ssiModel(:,1,i)), squeeze(ssiModel(:,2,i)), markers{1}, 'color', colors(1,:), ...
         'linewidth', 2, 'markersize', 4,'LineSmoothing','on');
      %linear least-squares fit
      fitParams(i,:) = polyfit(squeeze(ssiModel(:,1,i)), squeeze(ssiModel(:,2,i)),1);
   
      formatPlot(setupDir, colors,  markersLine, markers, noSize, labelSize, true, fitParams(i,:), i);
   end  
   
   %In debug mode, saving the figure and plotting the population response is not necessary
   if(debugOn)
      return; 
   end
   
   %% Saving
   fprintf('Saving...\n');
   if(~stdOn)
      name = sprintf('%s/fig3cde.png', outputdir);
   else
      name = sprintf('%s/fig3cde.png', outputdirStd);
   end
   print(name, '-dpng');
end


%% Function to format the plot nicely
function formatPlot(setupDir, colors, markersLine, markers, noSize, labelSize, isBottom, fitParams, i)
      
   box on;
   axis([-1.05,1.05,-1.05,1.05]);
   if(isBottom)
      xlabel('Selectivity', 'fontsize', labelSize);
   end
   if(i==1)
      ylabel('Sensory interaction', 'fontsize', labelSize);
   end

   set(gca, 'YTick', [-1,0,1]);
   set(gca, 'XTick', [-1,0,1]);

   %add linear least-squares fit to plot
   h = refline(fitParams(1,:));
   set(h,'Color', colors(1,:),'linewidth', 3, 'LineSmoothing','on');
   
   %legend
   if(~isBottom)
      text(-0.46,-0.88, sprintf(' Slope=%0.2f ', fitParams(1,1)), 'fontsize', noSize, 'EdgeColor', 'k', 'linewidth', 1);
   
   else
      
      switch(i)
      case 1
         condStr = 'Att. Away';
      case 2
         condStr = 'Att. Ref';
      case 3
         condStr = 'Att. Probe';
      end
      
      text(-0.46,-0.51, sprintf(' Slope=%0.2f ', fitParams(1,1)), 'fontsize', noSize, 'EdgeColor', 'k', 'linewidth', 1);
      legend({condStr}, 'location', 'southeast');  
   end

   set(gca, 'linewidth', 2, 'fontsize', noSize);
end
