%% / Experimental setup for competition of overlapping stimuli
% @author Frederik Beuth
%
% Data: MacEvoy, Tucker, Fitzpatrick (2009), Fig. 2a-c    
%
%
%% @section X3d1 Significant results (p. 2 bottom)
% - Significance test were not used. They simply stated that changing the angle between the gratings changes the tuning curve, 
%   leading to an unimodal tuning curve for a narrow angle.
%
%% @section X3d2 Setup
% - Stimuli are oriented bars at 1) 90 and 135°, 2) 70 and 110° and 3) 80 and 100°, so execute 3 subexperiments
% - Use 37 neurons to match each 5° of the range from 0° to 180° to a single neuron 
% - Record the neuron preferring the stimulus 1 in V1
% - Experiment does not use attention, so record the neurons only in the attend away conditions
% - Experiment uses overlapping stimuli, so place both stimuli at the same place
% - Experiment repeats the stimulus (0.25° width bar) every 3°, thus those drive adjacent neurons outside the RF too. 
%   We model 3 of these stimulus repetition via a large spatial dimension (6.6°) with 3
%   stimuli at 0.2°, 3.2, 6.4°.
%
%% @section X3d3 Calibration of the fit:
% - Use tuning curves as shown in MacEvoy2009, Fig2a-c
% - Calibrate the amount and shape of suppression afterwards
%
%\
%> @param setupDir      Directory with experimental setup, legend figures and a mat-file containing all plotting parameters
%> @param debugOn       If true, display additional information for fitting the data
%> @param stdOn         If true, run model with standard parameters
function figure3b(setupDir, debugOn, stdOn)

   if(nargin<1)
      setupDir = 'setups/';
      debugOn = false;
      stdOn = false;
   end
   load(sprintf('%s/plotParams.mat', setupDir));

   %% Run Model
   % Conditions: 
   % - row 1 => preferred stimulus alone
   % - row 2 => pair with attend away
   % - row 3 => anti-preferred stimulus alone   
   % - col 1 => complete different stimuli (45 and 135° oriented edges)
   % - col 1 => 50% different stimuli (70° and 110° oriented edges)
   % - col 3 => similar stimuli (80° and 110° oriented edges)
   if(~stdOn) 
      modelFile = sprintf('%s/modelMacevoy2009.mat', modelDir);
   else
      modelFile = sprintf('%s/modelMacevoy2009.mat', modelDirStd);
   end
   
   C = 3;             % # Conditions
   N = 3;             % # Subexperiments
   nGlobal = N*C;     % # All experiments
   L = 37;            % # Different stimuli/features
   X = 83;            % # Locations, correspond to 6.6° 
   Xrf = 5;           % Receptive field size, correspond to 0.4°
   Xstim = 3;         % Stimulus size, correspond to 0.25°
   x1 = [2:4,40:42,80:82];    % Stimulus locations in L4
   x3 = 21;           % Receptive field center in L2      


   %Model parameter (can be changed after creation)
   objModel = initModel(L,X,Xrf);
   if(~stdOn)  
      objModel.vSupFeat = 3.5;  
   end

   allIn = cell(nGlobal,1);
   allL4 = cell(nGlobal,1);
   allL2 = cell(nGlobal,1);
   allFEF = cell(nGlobal,1);
   allPFC = cell(nGlobal,1);  
   rModel = zeros(C,N);

   for s1 = 1:N

      % Stimulus preference at location 1(l1) and 2(l2)
      % - preferred stimulus is placed at l1
      % - anti-preferred stimulus is placed at l2

      switch(s1)
      case 1
         deg1 = 45;
         deg2 = 135;
      case 2
         deg1 = 70;
         deg2 = 110;
      case 3
         deg1 = 80;
         deg2 = 100;
      end

      l1 = round(deg1/180*(L-1))+1;
      l2 = round(deg2/180*(L-1))+1;

      %generate input population responses
      c = 8/L;
      g = 0.15;
      r1 = symetricalExpCircle(L, l1, g, 1, c);
      r2 = symetricalExpCircle(L, l2, g, 1, c);


      for c = 1:C
         s = (s1-1)*C+c; %global experiment index            
         rIn = zeros(L,X);

         %set 3x input stimulus
         if(c==1)
            rIn(:,x1) = repmat(r1, [1,3*Xstim]);  
         elseif(c==2)
            rIn(:,x1) = repmat(max(r1,r2), [1,3*Xstim]); 
         elseif(c==3)
            rIn(:,x1) = repmat(r2, [1,3*Xstim]);
         end            
         allIn{s} = rIn;

         % Set amplifications
         allFEF{s} = zeros(1,X);
         allPFC{s} = zeros(L,1); 

         %% Run model calculation
         [rL4, rL2] = calculateModel(allIn{s}, allFEF{s}, allPFC{s}, objModel);
         allL4{s} = rL4;
         allL2{s} = rL2;

         rModel(c,s1) = rL2(l1,x3);
      end % [conditions]

   end % [experiments]

   %normalize rModel like data to a maximum of 1
   rMax = max(rModel(:));
   if(debugOn)
      fprintf('Maximal model reponse=%0.3f\n', rMax);
   end
   rModel = rModel./rMax;

   save(modelFile, 'rModel', 'allIn', 'allFEF', 'allPFC', 'allL4', 'allL2', 'C', 'N', 'L', 'nGlobal');
   
   %% Replicate full tuning curves (MacEvoy2009, Fig. 1) for verification and debug
   if(debugOn)
      figure
      ysub = 1;
      xsub = 3;

      subplot(ysub, xsub, 1);
      plot([allL2{1}(:,2), allL2{2}(:,2), allL2{3}(:,2)]);
      ylim([0,1]);

      subplot(ysub, xsub, 2);
      plot([allL2{4}(:,2), allL2{5}(:,2), allL2{6}(:,2)]);
      ylim([0,1]);

      subplot(ysub, xsub, 3);
      plot([allL2{7}(:,2), allL2{8}(:,2), allL2{9}(:,2)]);
      ylim([0,1]);
   end
   
   
   %% Create plot for model
   figure
   hold on;
   
   plot(rModel(1,:), markersLine{1}, 'linewidth', 3, 'color', colors(1,:), 'markersize', 8, ...
      'LineSmoothing','on');
   plot(rModel(2,:), markersLine{3}, 'linewidth', 3, 'color', colors(3,:), 'markersize', 8, ...
      'LineSmoothing','on');
   plot(rModel(3,:), markersLine{5}, 'linewidth', 3, 'color', colors(5,:), 'markersize', 8, ...
      'LineSmoothing','on');
   
   formatPlot(setupDir, colors,  markersLine, markers, noSize, labelSize, true);
   
   %In debug mode, saving the figure and plotting the population response is not necessary
   if(debugOn)
      return; 
   end
   
   %% Saving
   fprintf('Saving...\n');
   if(~stdOn)
      name = sprintf('%s/fig3b.png', outputdir);
   else
      name = sprintf('%s/fig3b.png', outputdirStd);
   end
   print(name, '-dpng');
end


%% Function to format the plot nicely
function formatPlot(setupDir, colors,  markersLine, markers, noSize, labelSize, isBottom)

   ylim([0,1.05]);
   xlim([0.6, 3.4]);
   box on   
   
   ylabel('Response', 'fontsize', labelSize);
   set(gca, 'YTick', [0,1]);
   set(gca, 'XTick', [1,2,3]);
   set(gca, 'XTicklabel', []);
   set(gca, 'linewidth', 2, 'fontsize', noSize);

end
