%% / Experimental setup for biased competition and pooling with spatial attention
% @author Frederik Beuth
%
% Data: Lee and Maunsell (2010, J Neurosc), Fig. 5
%
%% @section X3c1 Significant results (p. 3061 right middle)
% Significance tests were not conducted, but the authors reported the following results:
% - Response to a pair of null and preferred stimulus is between the response of each individually.
% - Weak attentional modulation for a single stimulus (9% increase on average), strong for a stimulus pair.
% - Shifting attention from null to preferred stimulus of the pair increase the response on average by 59% (for preferred to null).
% - From attend away to attend preferred stimulus of the pair increase the responses by 28% (away to preferred)
% - Attention modulation was weak for 50-125ms: 18% for preferred to null, 10% for away to preferred
% - Attention modulation was strong for 125-250ms: 111% for preferred to null, 47% for away to preferred
%
%% @section X3a2 Setup
% - Stimuli are oriented moving gratings at 1) 90 and 270°, 2) 90 and 180° and 3) 90 and 90°, so execute 3 subexperiments
% - Record the neuron preferring stimulus 1 (assuming 90° orientation) in V4
% - Record neuron in all of the 4 conditions in each experiment
% - Contrary to Fig. 2, we use the data of a single stimulus in the attended own condition here(Fig. 2 uses attended away). 
%   This will result in slightly different parameters.
% - Stimuli have the size of about 0.7°-1.2° (Gabors with SD of 0.35°-0.6°) at unknown eccentricity. However, this size is 
%   very small compared to MT rf sizes at all eccentricities (4°-19° according to Lee and Maunsell), so we assume that a 
%   stimulus falls only in a single receptive field.
%
%% @section X3a3 Calibration of the fit:
% - Pretty standard biased competition, except a baseline has to be used as the data contains a peak after 
%   stimulus onset which is not modulated by attention.
%
%\
%> @param setupDir      Directory with experimental setup, legend figures and a mat-file containing all plotting parameters
%> @param debugOn       If true, display additional information for fitting the data
%> @param stdOn         If true, run model with standard parameters
function figure3a(setupDir, debugOn, stdOn)

   if(nargin<1)
      setupDir = 'setups/';
      debugOn = false;
      stdOn = false;
   end
   load(sprintf('%s/plotParams.mat', setupDir));

   %% Run Model
   % Conditions in each column: 
   % - 1 => preferred stimulus alone attended
   % - 2 => pair with attend own
   % - 3 => pair with attend away
   % - 4 => pair with attend other
   
   % Conditions in each row: 
   % - 1 => stimuli with 180° differing motion ('biased competition')
   % - 2 => stimuli with 90° differing motion
   % - 3 => stimuli with 0° differing motion ('feature pooling') 
   
   if(~stdOn)
      modelFile = sprintf('%s/modelLee2010_3a.mat', modelDir);
   else
      modelFile = sprintf('%s/modelLee2010_3a.mat', modelDirStd);
   end
         
   C = 4;             % # Conditions
   N = 3;             % # Subexperiments
   nGlobal = N*C;     % # All experiments
   L = 33;            % # Different stimuli
   X = 9;             % # Locations
   Xrf = 5;           % Receptive field size
   x1 = 4;            % Stimulus location 1 in L4
   x2 = 6;            % Stimulus location 2 in L4
   x3 = 3;            % Receptive field center in L2

   %Model parameter (can be changed after creation)
   objModel = initModel(L,X,Xrf);
   if(~stdOn)
      objModel.vSP = 1;
      objModel.vSupFeat = 2.75;
   end

   allIn = cell(nGlobal,1);
   allL4 = cell(nGlobal,1);
   allL2 = cell(nGlobal,1);
   allFEF = cell(nGlobal,1);
   allPFC = cell(nGlobal,1);  
   rModel = zeros(C,N);

   for s1 = 1:N

      % Stimulus preference at location 1 (l1) and 2 (l2)
      % - preferred stimulus is place at l1
      % - anti-preferred stimulus is placed at (l2) 
      l1 = round((L-1)*0.25)+1;
      switch(s1)
      case 1
         l2 = round((L-1)*0.75)+1;
      case 2
         l2 = round((L-1)*0.5)+1;
      case 3
         l2 = round((L-1)*0.25)+1;
      end

      %generate input population responses
      c = 6/L;
      r1 = symetricalExpCircle(L, l1, 0, 1, c); 
      r2 = symetricalExpCircle(L, l2, 0, 1, c);

      for c = 1:C
         s = (s1-1)*C+c; %global experiment index            

         % Set input
         if(c==1)
            rIn = zeros(L,X);
            rIn(:,x1) = r1;  
         else
            rIn = zeros(L,X);
            rIn(:,x1) = r1;  
            rIn(:,x2) = r2;
         end            
         allIn{s} = rIn;

         % Set amplification
         allFEF{s} = zeros(1,X);
         allPFC{s} = zeros(L,1); 
         if(c==1 || c==2)
            allFEF{s}(x1) = 1;
         end
         if(c==4)
            allFEF{s}(x2) = 1;
         end            

         %% Run model calculation
         [rL4, rL2] = calculateModel(allIn{s}, allFEF{s}, allPFC{s}, objModel);
         allL4{s} = rL4;
         allL2{s} = rL2;

         rModel(c,s1) = rL2(l1,x3);
      end % [conditions]

   end % [experiments]
   %normalize maximal response to 1.0 like in the data
   maxModel = max(rModel(:));
   rModel = rModel./maxModel;

   save(modelFile, 'rModel', 'allIn', 'allFEF', 'allPFC', 'allL4', 'allL2', 'C', 'N', 'L', 'nGlobal');
   
   
   %% Create plot for model 
   figure
   hold on;   
   for i=1:4
      plot(rModel(i,:), markersLine{i}, 'linewidth', 3, 'color', colors(i,:), 'markersize', 8, ...
         'LineSmoothing','on');
   end   
   formatPlot(setupDir, colors,  markersLine, markers, noSize, labelSize, true);
   
   %In debug mode, saving the figure and plotting the population response is not necessary
   if(debugOn)
      return; 
   end
   
   %% Saving
   fprintf('Saving...\n');
   if(~stdOn)
      name = sprintf('%s/fig3a.png', outputdir);
   else
      name = sprintf('%s/fig3a.png', outputdirStd);
   end
   print(name, '-dpng');
end


%% Function to format the plot nicely
function formatPlot(setupDir, colors,  markersLine, markers, noSize, labelSize, isBottom)

   ylim([0,1.05]);
   xlim([0.6, 3.4]);
   box on   
   
   ylabel('Response', 'fontsize', labelSize);
   set(gca, 'YTick', [0,1]);
   set(gca, 'XTick', [1,2,3]);
   set(gca, 'XTicklabel', []);
   set(gca, 'linewidth', 2, 'fontsize', noSize);
   
end
