%% / Experimental setup for biased competition  with spatial attention
% @author Frederik Beuth
%
% Data: Lee and Maunsell (2010, J Neurosc), Fig. 5
%
%% @section X4a1 Significant results (p. 3061 right middle)
% - See figure3a.m
%
%% @section X4a2 Setup
% - Stimuli are oriented moving gratings at 1) 90° and 270°, 2) 90° and 180°, and 3) 90° and 90°, so execute 3 subexperiments
% - Record the neuron preferring stimulus 1 (assuming 90° orientation) in V4
% - Record neuron in all of the 5 conditions in each subexperiment
%
%% @section X4a3 Calibration of the fit
% - Pretty standard biased competition, except:
% - A baseline has to be used as the data contains a peak after stimulus onset which is not 
%   modulated by attention.
%
%\
%> @param setupDir      Directory with experimental setup, legend figures and a mat-file containing all plotting parameters
%> @param debugOn       If true, display additional information for fitting the data
%> @param stdOn         If true, run model with standard parameters
function figure2(setupDir, debugOn, stdOn)

   if(nargin<1)
      setupDir = 'setups/';
      debugOn = false;
      stdOn = false;
   end
   load(sprintf('%s/plotParams.mat', setupDir));

   %% Run Model
   % Conditions in each column: 
   % - 1 => preferred stimulus alone (attend away)
   % - 2 => anti-preferred stimulus alone (attend away)
   % - 3 => pair with attend own
   % - 4 => pair with attend away
   % - 5 => pair with attend other
   
   % Conditions in each row: 
   % - 1 => stimuli with 180° differing motion ('biased competition')
   % - 2 => stimuli with 90° differing motion
   % - 3 => stimuli with 0° differing motion ('feature pooling')   
   
   % General parameter
   if(~stdOn)
      modelFile = sprintf('%s/modelLee2010_2.mat', modelDir);
   else
      modelFile = sprintf('%s/modelLee2010_2.mat', modelDirStd);
   end
   C = 5;             % # Conditions
   N = 1;             % # Subexperiments
   nGlobal = N*C;     % # All experiments
   L = 33;            % # Different stimuli
   X = 9;             % # Locations
   Xrf = 5;           % Receptive field size
   x1 = 4;            % Location of stimulus 1 in L4
   x2 = 6;            % Location of stimulus 2 in L4
   x3 = 3;            % Receptive field center in L2
   
   % Model parameter (can be changed after creation)
   objModel = initModel(L,X,Xrf);
   if(~stdOn)
      objModel.vSupFeat = 2.5;
      objModel.vSP = 2;
      objModel.baseline = 0.15;
   end
   
   % Store all responses for all conditions
   allIn = cell(nGlobal,1);
   allL4 = cell(nGlobal,1);
   allL2 = cell(nGlobal,1);
   allFEF = cell(nGlobal,1);
   allPFC = cell(nGlobal,1);
   
   % place stimulus at location 1 (l1) and 2 (l2)
   % preferred (l1) and anti-preferred stimulus (l2)
   l1 = round((L-1)*0.25)+1;
   l2 = round((L-1)*0.75)+1;
   
   % generate input population responses
   c = 6/L;
   g = 0.0;
   r1 = symetricalExpCircle(L, l1, g, 1, c);
   r2 = symetricalExpCircle(L, l2, g, 1, c);
   
   for c = 1:C
      s = c;%global experiment index
      
      %set input
      if(c==1)
         rIn = zeros(L,X);
         rIn(:,x1) = r1;
      elseif(c==2)
         rIn = zeros(L,X);
         rIn(:,x2) = r2;
      else
         rIn = zeros(L,X);
         rIn(:,x1) = r1;
         rIn(:,x2) = r2;
      end
      allIn{s} = rIn;
      
      % Set amplifications
      allFEF{s} = zeros(1,X);
      allPFC{s} = zeros(L,1);
      if( c==3)
         allFEF{s}(x1) = 1;
      end
      if(c==5)
         allFEF{s}(x2) = 1;
      end
      
      allIn{s} = rIn;
      
      %% Run model calculation
      [rL4, rL2] = calculateModel(allIn{s}, allFEF{s}, allPFC{s}, objModel);
      allL4{s} = rL4;
      allL2{s} = rL2;
      
   end % [conditions]
   
   save(modelFile, 'allIn', 'allFEF', 'allPFC', 'allL4', 'allL2', 'X', 'L', 'C', 'N', 'nGlobal', 'x1', 'x2', 'x3', 'l1', 'l2');
     

   %% Create plot for model
   % dim1=1 => pair with attend away  ( condition 1 in Lee2010)
   % dim1=2 => pair with attend own   ( condition 2 in Lee2010)
   % dim1=3 => pair with attend other ( condition 3 in Lee2010)   
   figure
   rModel = [allL2{1}(l1,x3), allL2{2}(l1,x3), allL2{4}(l1,x3), allL2{3}(l1,x3), allL2{5}(l1,x3); ...
        allL2{1}(l2,x3), allL2{2}(l2,x3), allL2{4}(l2,x3), allL2{3}(l2,x3), allL2{5}(l2,x3)]';
   bar(rModel, 0.9, 'FaceColor', colors(6,:), 'EdgeColor', 'none');   
   formatPlot(setupDir, colors,  markersLine, markers, noSize, labelSize, true, size(rModel,1));
     
   %In debug mode, saving the figure and plotting the population response is not necessary
   if(debugOn)
      return; 
   end
   
   %% Saving
   fprintf('Saving...\n');
   figId = 2;
   if(~stdOn)
      name = sprintf('%s/fig%i.png', outputdir, figId);
   else
      name = sprintf('%s/fig%i.png', outputdirStd, figId);
   end
   print(name, '-dpng');
   
end

%% Function to format the plot nicely
function formatPlot(setupDir, colors,  markersLine, markers, noSize, labelSize, isBottom, L)

   ylim([0,1.05]);
   xlim([0.3,L+0.7]);
   box off
   
   nX = 5;
   ylabel('Response', 'fontsize', labelSize);
   set(gca, 'YTick', [0,1]);
   set(gca, 'XTick', linspace(1,L,nX));
   if(isBottom)
      set(gca, 'XTicklabel', 1:5);
   else
      set(gca, 'XTicklabel', []);
   end
   set(gca, 'linewidth', 2, 'fontsize', noSize, 'Layer', 'top');
      
end
