%% / Evaluate spatial pooling in the normalization model of attention 
% @author Frederik Beuth
%
% - Model reference: Reynolds and Heeger (2009)
% - Experiment: biased competition with spatial attention
% - Experiment reference: Reynolds, Chelazzi, Desimone (1999)
% - Intended for comparing the model of Reynolds & Heeger (2009) with our model (see figure16b.m)
%
%\
%> @param debugOn       If true, display additional information for fitting the data
function figure11d(debugOn)

   if(nargin<1)
      debugOn = false;
   end
   % Add path to sourcecode of model
   addpath('modelReynolds2009');
   
   %% Parameters  
   % See description of Reynolds & Heeger (2009)
   stimWidth = 5;
   AxWidth = 5;
   Apeak = 5;
   rfSize = 51; % should be odd
   
   % We assume that spatial amount of suppression is equal to receptive field size. 
   % As the suppression is modeled via a 1D Gaussian which should have circa 50% maximal activity 
   % at the receptive field borders, sigma of the Gaussian should be roughly rfSize/2.5
   IxWidth = rfSize/2.5; 
   
   % Calibrate stimuli
   stimCenter1 = 90;
   stimCenter2 = 110;
   att_away_loc = -100;
   RF_center = round(mean([stimCenter1,stimCenter2]));
   
   stimOrientation1 = -90;
   stimOrientation2 = 90;
      
   % Set contrast to maximum (1.0)
   contrast = 1;
  
   %% Run model
   % Sampling of space and orientation
   x = [-200:200];
   theta = [-180:180]';      
   
   rfH = (rfSize-1)/2; % half of receptive field 
   
   % Create stimuli
   stim1 = contrast * makeGaussian(theta,stimOrientation1,1,1) * makeGaussian(x,stimCenter1,stimWidth,1);
   stim2 = contrast * makeGaussian(theta,stimOrientation2,1,1) * makeGaussian(x,stimCenter2,stimWidth,1);
   pair = stim1 + stim2;

   % Population response when attending away 
   % with adding  " 'showActivityMaps',1,'showModelParameters',1", you can inspect the model
   Pair_att_away = attentionModel(x,theta,pair,'IxWidth', IxWidth, ...
      'Apeak',Apeak,'Ax',att_away_loc,'AxWidth',AxWidth);
   
   % Population response when attending location 1 containing stim 1 
    Pair_att_loc1 = attentionModel(x,theta,pair, 'IxWidth', IxWidth, ...
      'Apeak',Apeak, 'Ax',stimCenter1,'AxWidth',AxWidth);
   
   % Population response when attending location 2 containing stim 2 
   Pair_att_loc2 = attentionModel(x,theta,pair, 'IxWidth', IxWidth, ...
      'Apeak',Apeak,'Ax',stimCenter2,'AxWidth',AxWidth);
   

   %% Plot responses over space
   % Pick neurons to record (feature l1 or l2, locations x1 to x2)
   l1 = find(theta==stimOrientation1);
   l2 = find(theta==stimOrientation2);
   x1 = find(x==(RF_center-rfH));
   x2 = find(x==(RF_center+rfH));
   
   %% Debug
   if(debugOn)
      figure
      subplot(2,1,1);
      plot([-rfH:rfH], [Pair_att_away(l1,x1:x2); ...
         Pair_att_loc1(l1,x1:x2); ...
         Pair_att_loc2(l1,x1:x2)]');
      xlim([-rfH,rfH]);
      
      title('Response of cells encoding stimulus 1');
      xlabel('Location within RF');
      ylabel('Reponse');      
      legend({'Attend away', ...
         'Attend loc. 1' ...
         'Attend loc. 2'}, 'location', 'northeastoutside');
      
      subplot(2,1,2);
      plot([-rfH:rfH], [Pair_att_away(l2,x1:x2); ...
         Pair_att_loc1(l2,x1:x2); ...
         Pair_att_loc2(l2,x1:x2);]');
      xlim([-rfH,rfH]);
      
      title('Stimulus 2');
      xlabel('Location within RF');
      ylabel('Reponse');      
      legend({'Attend away', ...
         'Attend loc. 1' ...
         'Attend loc. 2'}, 'location', 'northeastoutside');
   end   

   %% Plot model
   load('setups/plotParams.mat');
   sizeFig = [0.5, 0.5, 11, 14];
   figure('paperposition', sizeFig);
   addpath('strings/');
   plotPositions = [0.18, 0.5, 0.62, 0.21; ...
                    0.18, 0.07, 0.62, 0.21];
   labelSize = 20;
   noSize = 16;
   
   subplot('position', plotPositions(1,:));
   hold on;
      
   maxY = max(max(Pair_att_away(:)), max(Pair_att_loc1(:)));
   plot(Pair_att_away(l1,x1:x2), '-', 'linewidth', 3, 'color', colors(1,:), ...
      'LineSmoothing','on');
   plot(Pair_att_loc1(l1,x1:x2), '-', 'linewidth', 3, 'color', colors(2,:), ...
      'LineSmoothing','on');
   formatPlot( noSize, labelSize, false, rfSize+1, maxY, rfH);
   
   % Plot input
   subplot('position', plotPositions(2,:));
   hold on;
   
   plot(pair(l1,x1:x2), '-', 'linewidth', 3, 'color', colors(1,:), ...
      'LineSmoothing','on');
   plot(pair(l2,x1:x2), '--', 'linewidth', 3, 'color', colors(4,:), ...
      'LineSmoothing','on');
   formatPlot( noSize, labelSize, true, rfSize+1, 1, rfH);
   
   %% Saving
   fprintf('Saving...\n');
   name= sprintf('%s/fig11d.png', outputdir);
   print(name, '-dpng');
   
end


%% Function to format the plot nicely
function formatPlot( noSize, labelSize, isBottom, X, maxY, rfH)

   xlim([1,X]);
   ylim([0, maxY*1.05]);
   box off   
   
   labelX = [-20, -10, 0, 10, 20];
    ylabel('Response', 'fontsize', labelSize);
   set(gca, 'YTick', [0,ceil(maxY)]);
   set(gca, 'XTick', labelX + rfH+1);
   if(~isBottom)
      title('Model', 'fontsize', labelSize);
      set(gca, 'XTicklabel', '');
      hl=legend({'Att. away', 'Att. own'}, 'location', 'northeast', 'fontsize', labelSize);
      set(hl, 'position', get(hl, 'position')+ [0.13,0,0,0]); 
   else
      title('Input', 'fontsize', labelSize);
      set(gca, 'XTicklabel', labelX);
      hl=legend({'Stim 1', 'Stim 2'}, 'location', 'northeast', 'fontsize', labelSize);  
      set(hl, 'position', get(hl, 'position')+ [0.14,0,0,0]); 
   end
  
   set(gca, 'linewidth', 2, 'fontsize', noSize);
   
end
