%% / Evaluate spatial pooling in our microcircuit model of attention 
% @author Frederik Beuth
%
% - Experiment: biased competition with spatial attention
% - Data: Reynolds, Chelazzi, Desimone (1999).
% - Intended for a direct comparison with the model of Reynolds & Heeger (2009): see figure16d.m
%
%\
%> @param setupDir      Directory with experimental setup, legend figures and a mat-file containing all plotting parameters
%> @param debugOn       If true, display additional information for fitting the data
%> @param stdOn         If true, run model with standard parameters
function figure11b(setupDir, debugOn, stdOn)

   if(nargin<1)
      setupDir = 'setups/';
      debugOn = false;
      stdOn = false;
   end
   load(sprintf('%s/plotParams.mat', setupDir));

   
   %% Run model
   %conditions: 7-9
   % 7 => pair attend away
   % 8 => pair attend reference
   % 9 => pair attend probe

   C = 9;            % # Conditions
   N = 1;            % # Subexperiments
   nGlobal = N*C;    % # All experiments
   L = 17;           % # Different stimuli/features

   X = 801;           % # Locations (identical chosen as in Reynolds2009)
   Xrf = 51;          % Receptive field size
   x1 = 400-10;       % Stimulus location 1
   x2 = 400+10;       % Stimulus location 2
   x3 = 17;           % Receptive field center
   rfHL4 = (Xrf-1)/2; %

   %Model parameter (can be changed after creation)
   objModel = initModel(L,X,Xrf);
   if(~stdOn)
      objModel.vSupFeat = 3;
      objModel.pSupFeat = 2;
      objModel.vSP = 5;
   end

   allIn = cell(nGlobal,1);
   allL4 = cell(nGlobal,1);
   allL2 = cell(nGlobal,1);
   allFEF = cell(nGlobal,1);
   allPFC = cell(nGlobal,1);
   
   for s1 = 1:N

      % Choose orientation/color of reference (l1) and probe stimulus (l2) in the feature space 
      % - Here arbitrary chosen as 50% difference, implying 25% difference in feature space
      l1 = (L-1)*(2/8); 
      l2 = (L-1)*(6/8);

      % Generate input 
      % Gauss-function: provides an identical input as in Reynolds and Heegers (2009) model
      sigmaL = L/10;
      sigmaX = 5;
      r1 = customgauss([L,X], sigmaL, sigmaX, 0, 0, 1, [l1-(L-1)/2-1, x1-(X-1)/2-1]);
      r2 = customgauss([L,X], sigmaL, sigmaX, 0, 0, 1, [l2-(L-1)/2-1, x2-(X-1)/2-1]);
      a1 = gauss1D(X, x1, sigmaX);
      a2 = gauss1D(X, x2, sigmaX);      

      for c = 7:C
         s = (s1-1)*C+c; %global experiment index 

         % Set input
         %reference or probe alone
         if(c<=6)
            error('Condition not used here');
         %pair
         elseif(c<=9)
            rIn = max(r1, r2);
            allIn{s} = rIn;
         else
            allIn{s} = allIn{s-1};
         end            

         % Set amplifications
         allFEF{s} = zeros(1,X);
         allPFC{s} = zeros(L,1); 
         if(c==2 || c==5 || c==8)
            allFEF{s} = a1;
         end
         if(c==3 || c==6 || c==9)
            allFEF{s} = a2;
         end            

         %% Run model calculation
         [rL4, rL2] = calculateModel(allIn{s}, allFEF{s}, allPFC{s}, objModel);
         allL4{s} = rL4;
         allL2{s} = rL2;
      end
   end %[experiments]

   
   %% Plot responses over space
   % Pick neurons to record (feature l1 or l2, locations x1:x2)  
   % location in L4 indices
   Xhalf = (X-1)/2;
   x1L4 = Xhalf-rfHL4;
   x2L4 = Xhalf+rfHL4;
   % same location in L2 indices, manually slightly adapted (+2) to record the correct cell
   rfspaceL2 = round((objModel.resL2(2)-1)/(objModel.resL4(2)-1) * Xrf) + 1 + 2;
   rfHL2 = (rfspaceL2-1)/2;
   x1L2 = x3-rfHL2;
   x2L2 = x3+rfHL2;
   
   load('setups/plotParams.mat');
   sizeFig = [0.5, 0.5, 11, 14];
   figure('paperposition', sizeFig);
   addpath('strings/');
   plotPositions = [0.15, 0.70, 0.65, 0.21;  ...
                    0.15, 0.39, 0.65, 0.21;  ...
                    0.15, 0.07, 0.65, 0.21];
   labelSize = 20;
   noSize = 16;
   
   % Plot model reponses - L2
   subplot('position', plotPositions(1,:));
   hold on;
   plot(allL2{7}(l1,x1L2:x2L2), '-', 'linewidth', 3, 'color', colors(1,:), ...
      'LineSmoothing','on');
   plot(allL2{8}(l1,x1L2:x2L2), '-', 'linewidth', 3, 'color', colors(2,:), ...
      'LineSmoothing','on');
   formatPlot( noSize, labelSize, 1, rfspaceL2, 1, rfHL2);
   
   % Plot model reponses - L4
   subplot('position', plotPositions(2,:));
   hold on;
   plot(allL4{7}(l1,x1L4:x2L4), '-', 'linewidth', 3, 'color', colors(1,:), ...
      'LineSmoothing','on');
   plot(allL4{8}(l1,x1L4:x2L4), '-', 'linewidth', 3, 'color', colors(2,:), ...
      'LineSmoothing','on');
   formatPlot( noSize, labelSize, 2, Xrf, 1, rfHL4);
   
   % Plot reponses - input
   subplot('position', plotPositions(3,:));
   hold on;
   plot(allIn{7}(l1,x1L4:x2L4), '-', 'linewidth', 3, 'color', colors(1,:), ...
      'LineSmoothing','on');
   plot(allIn{7}(l2,x1L4:x2L4), '--', 'linewidth', 3, 'color', colors(4,:), ...
      'LineSmoothing','on');
   formatPlot( noSize, labelSize, 3, Xrf, 1, rfHL4);
   
   %% Saving
   fprintf('Saving...\n');
   if(~stdOn)
      name = sprintf('%s/fig11b.png', outputdir);
   else
      name = sprintf('%s/fig11b.png', outputdirStd);
   end
   print(name, '-dpng');  
end


%% Function to format the plot nicely
function formatPlot( noSize, labelSize, plotId, X, maxY, rfHL4)

   xlim([1,X]);
   ylim([0, maxY*1.05]);
   box off   
   
   labelX = [-20, -10, 0, 10, 20];
    ylabel('Response', 'fontsize', labelSize);
   set(gca, 'YTick', [0,ceil(maxY)]);
   set(gca, 'XTick', labelX + rfHL4+1);
   if(plotId <=2)
      if(plotId==1)
         title('Layer 2/3', 'fontsize', labelSize);
      else
         title('Layer 4', 'fontsize', labelSize);
      end
      set(gca, 'XTicklabel', '');
      hl = legend({'Att. away', 'Att. own'}, 'location', 'northeast', 'fontsize', labelSize);
      set(hl, 'position', get(hl, 'position')+ [0.12,0,0,0]); 
   else
      title('Input', 'fontsize', labelSize);
      set(gca, 'XTicklabel', labelX);
      hl=legend({'Stim 1', 'Stim 2'}, 'location', 'northeast', 'fontsize', labelSize);  
      set(hl, 'position', get(hl, 'position')+ [0.13,0,0,0]); 
   end
   set(gca, 'linewidth', 2, 'fontsize', noSize);
   
end
