%% / Experimental setup for biased competition and attentional modulation of surround suppression
% @author Frederik Beuth
%
% Data: This is a model prediction, so no data is available
%
%% @section X14b2 Setup:
%   - Combine the biased competition setup, containing a preferred and an anti-preferred stimulus inside the classical receptive field (cRF), 
%     with an anti-preferred stimulus in the surround
%   - Three stimuli: preferred inside cRF, anti-preferred inside cRF, anti-preferred in the surround. A real setup
%     should contain an additional stimulus for attend-away, which we do not simulate
%   - Use spatial attention as it is more common in biased competition. However, feature-based attention would be complicated
%     as two stimuli are similar
%   - Attention has the same size as the stimulus
%   - Stimuli inside cRF are chosen small to place two of them within the cRF, and to ensure that both stimuli fall only in 
%     a single V4 receptive field. 
%   - Surround stimulus is large to increase its influence.
%   - Contrast of all stimuli is maximal (1.0)
%
%% @section X14b1 Predicted significant results:
% - Attend away: Surround suppression decreases the response of neurons in the classical receptive field (cRF) with similar 
%   features, thus the neurons encoding the anti-preferred stimulus. 
%   This leads to a decreased feature-based suppression to neurons encoding the preferred stimulus, increasing the 
%   response of them. Thus it favors the competition between the stimuli inside the cRF for the other stimulus.
%
% - Attend preferred stimulus inside the cRF: Similar effect as in classical biased competition. The response of preferred encoding neurons
%   is amplified, which cancels out the feature-based suppression from the anti-preferred encoding neurons.
%
% - Attend  anti-preferred stimulus inside the cRF: Similar effect as in classical biased competition. The response of 
%   the anti-preferred encoding neurons is amplified, which diminishes feature-based as well as surround suppression at these 
%   neurons. Thus, anti-preferred encoding neurons are strongly active, suppressing the preferred encoding neurons.
%
% - Attend anti-preferred stimulus in the surround: The neurons in the surround are amplified, increasing suppression to the
%   anti-preferred encoding neurons. This diminished response eliminates the feature-based suppression to the preferred encoding 
%   neurons almost completely. So, the response of these neurons for this configuration is almost identical to the response
%   of the preferred stimulus being presented alone.
%
%
%% @section X14b3 Calibration of the fit:
%   - 1) Use standard values for all parameters
%   - 2) Increase size of surround stimulus to show a significant surround suppression effect 
%
%\
%> @param setupDir      Directory with experimental setup, legend figures and a mat-file containing all plotting parameters
%> @param debugOn       If true, display additional information for fitting the data
%> @param stdOn         If true, run model with standard parameters
function figure10(setupDir, debugOn, stdOn)

   if(nargin<1)
      setupDir = 'setups/';
      debugOn = false;
      stdOn = false;
   end
   load('setups/plotParams.mat');
   clear 'plotPositions';
   plotPositions{1} = [0.08, 0.74, 0.87, 0.25];
   plotPositions{2} = [0.08, 0.40, 0.87, 0.25];
   plotPositions{3} = [0.08, 0.06, 0.87, 0.25];
  
   %% Run Model
   if(~stdOn)
      modelFile = sprintf('%s/modelPrediction1.mat', modelDir);
   else
      modelFile = sprintf('%s/modelPrediction1.mat', modelDirStd);
   end
   % Setups/Conditions:
   % - 1/1 => preferred stimulus alone with attend away
   % - 1/2 => preferred stimulus alone with attend preferred
   % - 2/1 => pair with attend away
   % - 2/2 => pair with attend preferred
   % - 2/3 => pair with attend anti-preferred      
   % - 3/1 => pair and surround with attend away
   % - 3/2 => pair and surround with attend preferred
   % - 3/3 => pair and surround with attend anti-preferred      
   % - 3/4 => pair and surround with attend surround
   C = 3;             % # Setups
   N = 4;             % # Conditions
   nGlobal = N*C;     % # All experiments
   L = 33;            % # Different orientations
   X = 53;            % # Different locations
   Xrf = 5;           % Receptive field size
   Xstim = 1;         % Stimulus size inside cRF
   XstimSur = 15;     % Stimulus size of surround
   x1 = 44;           % Preferred stimulus location in L4
   x2 = 46;           % Anti-preferred stimulus location in L4
   x3 = 9;            % Surround stimulus location in L4      
   x4 = 45;           % Center of cRF in L4
   x5 = 23;           % Receptive field center in L2      
   x6 = 5;            % Receptive field of surround in L2      
   XH = (Xstim-1)/2;  % Half of stimulus
   XHsur = (XstimSur-1)/2;  % Half of surround stimulus

   %Model parameter
   objModel = initModel(L,X,Xrf);

   allIn = cell(nGlobal,1);
   allL4 = cell(nGlobal,1);
   allL2 = cell(nGlobal,1);
   allFEF = cell(nGlobal,1);
   allPFC = cell(nGlobal,1);        
   rModel = nan(C,N, 2);
   rModelSur = nan(C,N); % Record surround stimulus for debugging

   % Create feature tuning curves of the input
   % Stimulus preference of pref (l1), anti-pref (l2), and surround (l3)
   l1 = ((L-1)*0.25)+1;
   l2 = ((L-1)*0.75)+1;
   l3 = ((L-1)*0.75)+1;

   c = 8/L;
   g = 0.0; % baseline
   r1 = (symetricalExpCircle(L, l1, 0, 1, c) * (1-g) + g);
   r2 = (symetricalExpCircle(L, l2, 0, 1, c) * (1-g) + g);
   r3 = (symetricalExpCircle(L, l3, 0, 1, c) * (1-g) + g);
   r1 = repmat(r1, [1,Xstim]);
   r2 = repmat(r2, [1,Xstim]);
   r3 = repmat(r3, [1,XstimSur]);

   for s1 = 1:N
      for c = 1:C

         % Do not run model for conditions where attended stimulus is not present
         if( (c == 1 && (s1==3 || s1==4)) || (c == 2 && s1==4) )
            continue;
         end

         s = (s1-1)*C+c; %global experiment index            

         % Set input
         rIn = zeros(L,X);
         % Preferred stimulus 
         rIn(:,x1-XH:x1+XH) = r1;
         % Anti-preferred stimulus               
         if(c>1)
            rIn(:,x2-XH:x2+XH) = r2;               
         end            
         % Surround stimulus               
         if(c>2)
            rIn(:,x3-XHsur:x3+XHsur) = r3;               
         end            
         allIn{s} = rIn;

         % Set amplification
         allFEF{s} = zeros(1,X);
         allPFC{s} = zeros(L,1); 
         if(s1==2)
            allFEF{s}(x1-XH:x1+XH) = 1; 
         end
         if(s1==3)
            allFEF{s}(x2-XH:x2+XH) = 1; 
         end
         if(s1==4)
            allFEF{s}(x3-XHsur:x3+XHsur) = 1; 
         end                      

         %% Run model calculation
         [rL4, rL2] = calculateModel(allIn{s}, allFEF{s}, allPFC{s}, objModel);
         allL4{s} = rL4;
         allL2{s} = rL2;

         rModel(c,s1,1) = rL2(l1,x5);  % Preferred inside cRF
         rModel(c,s1,2) = rL2(l2,x5);  % Anti-preferred inside cRF
         rModelSur(c,s1) = rL2(l2,x6); % Anti-preferred surround 
      end % [conditions]
      fprintf('.');

   end % [experiments]      

   % normalize model response as data responses
   maxData = max(rModel(:));
   minData = 0;
   rModel = (rModel - minData) ./ (maxData - minData);
   if(debugOn)
      fprintf('\nMax. model response = %0.3f; min. response = %0.3f\n', maxData, minData);
      rModelSur = (rModelSur - minData) ./ (maxData - minData);
   end

   save(modelFile, 'rModel', 'allIn', 'allFEF', 'allPFC', 'allL4', 'allL2', 'C', 'N', 'L', 'nGlobal', ...
      'X', 'Xrf', 'x1', 'x2', 'x3', 'x4', 'x5', 'x6', 'l1', 'l2', 'l3', 'rModelSur');

   
   %% Create plot for model
   figure
   for i=1:C
      subplot('position', plotPositions{i});  
      bar(squeeze(rModel(i,:,:)), 0.5, 'FaceColor', colors(6,:), 'EdgeColor', 'none');
      isBottom = (i==3);
      formatPlot(setupDir, colors,  markersLine, markers, noSize, labelSize, isBottom, N);
   end % [i]
   
   % Plot surround recordings in debug mode
   if(debugOn)
      figure
      for i=1:C
         subplot('position', plotPositions{i});
         bar(rModelSur(i,:), 0.15, 'FaceColor', colors(6,:), 'EdgeColor', 'none');
         isBottom = (i==3);
         formatPlot(setupDir, colors,  markersLine, markers, noSize, labelSize, isBottom, N);
      end % [i]
   end
   
   %In debug mode, saving the figure and plotting the population response is not necessary
   if(debugOn)
      return; 
   end
      
   %% Saving
   fprintf('Saving...\n');
   if(~stdOn)
      name = sprintf('%s/fig10.png', outputdir);
   else
      name = sprintf('%s/fig10.png', outputdirStd);
   end
   print(name, '-dpng');   
   
end


%% Function to format the plot nicely
function formatPlot(setupDir, colors,  markersLine, markers, noSize, labelSize, isBottom, nX)
   
   ylim([0,1.05]);
   xlim([0.5,nX+0.5]);
   box off
   
   ylabel('Response', 'fontsize', labelSize);
   set(gca, 'YTick', [0,1]);
   set(gca, 'XTick', [1:nX]);
   set(gca, 'XTicklabel', []);
   if(~isBottom)
      set(gca, 'YTicklabel', []);
   end
   set(gca, 'linewidth', 2, 'fontsize', noSize);
   
end
