%% / Calculate the microcircuit model of attention
% @author Frederik Beuth
% 
% @mainpage
%% @section s1 Overview
% - This source code implements the microcircuit model of attention, described in the article:
%   Beuth F, Hamker FH (2015) "A mechanistic cortical microcircuit of attention
%   for amplification, normalization and suppression", Vision Research. 
%   Please cite this paper if this code is used in or to motivate any publications.
% - In the code, L denotes number of feature in layer 4 and layer 2/3, and 
%   X the number of locations in layer 4. Layer 2/3 has less space, denoted by Xpooled, due to spatial
%   pooling.
%
%% @section s2 Getting started
% - The files calculateModel.m and initModel.m implement the model, whereby the former executes
%   the calculations and the later creates a structure (called 'model object') containing all parameters.
%   The parameters are passed to calculateModel(..) via the model object. 
% - Parameters should be modified in the user function, hence outside of initModel.m. 
%   A description of all parameter can be found in the file initModel.m.
% - Thus, a user function should call the files typically in this order:
%   - 1) Create model structure: objModel = initModel(..)
%   - 2) Modify parameters, e.g. objModel.vSP = 5
%   - 3) Run model: calculateModel(.., objModel)
% - The files figure*.m reproduce the figures of the publication. They can be used as examples.
% - The file calculateModel.m can also be executed standalone as a toy-example. The example simulates 
%   two stimuli with spatial attention to the second one. 
%   They do not suppress each other, because they are not placed within a receptive field, so 
%   feature-based suppression is not occurring. They are different, so that surround suppression is also not
%   occurring.
%
%\
% $Rev: 555 $
% $Date: 2014-07-06 23:02:20 +0200 (So, 06. Jul 2014) $
% $Author: beuth $
%
%> @param input     Input from previous visual area (size: L x X)
%> @param aFeat     Feature-based amplification from a higher cortical area like PFC (size: L x 1)
%> @param aSpatial  Spatial amplification from FEF or LIP (size: 1 x X)
%> @param this      Model structure, containing all parameters
%> @param plotResponses   If true, plot the neuronal responses at converged state
%> @return rL4            Neuronal responses of layer 4 (size: L x X)
%> @return rL2            Neuronal responses of layer 4 (size: L x Xpooled)
%> @return supFeatRet     Feature-based suppression (size: L x X)
%> @return supSpatialRet  Surround suppression (size: L x Xpooled)
function [rL4, rL2, supFeatRet, supSpatialRet] = calculateModel(input, aSpatial, aFeat, this, plotResponses)

   if(nargin<5)
      plotResponses = false;
   end
   % Toy example
   if(nargin<1)
      this = initModel(2,9,3);
      input = [0,1,1,0,0,0,0,0,0;
               0,0,0,0,0,0,1,1,0]*0.3;
      aFeat = [0,0]';
      aSpatial = [0,0,0,0,0,0,1,0,0];
      plotResponses = true;
   end

   %% Check sizes
   if(size(aSpatial,2) ~= this.resL4(2))
      error('Spatial amplification has wrong dimension: %i. Expected was: %i', size(aSpatial,2), this.resL4(2));
   end
   if(size(aFeat,1) ~= this.resL2(1))
      error('Feature-based amplification has wrong dimension: %i. Expected was: %i', size(aFeat,1), this.resL2(1));
   end

   %% Preallocate memory for speed
   rL4 = zeros(this.resL4);                  % Rates in layer 4
   rL2 = zeros(this.resL2);                  % Rates in layer 2/3
   rFB = zeros(this.resL4);                  % Temporary variable for feedback from layer 2/3 to layer 4
   supFeat = zeros(this.resL4);              % Feature-based suppression from layer 2/3 to layer 4
   excL2 = zeros(this.resL2);                % Excitation for each neuron in layer 2/3 
   supSur = zeros(this.resL4);               % Surround suppression from layer 2/3 to layer 4
   
   rL4Ext = zeros(this.resL4+[0,2*this.rfH_FF]);   % rL4 and rL2 are extended by half receptive field size at each border
   rL2Ext = zeros(this.resL2+[0,2*this.rfH_FB]);
        
   % Pump up 1D-amplification vectors to the same 2D-size as each layer. For speed reasons, do it before the loop.
   aSpatial_2D = repmat(aSpatial, this.resL4(1), 1) .* this.vSP;
   aFeat_2D = repmat(aFeat, 1, this.resL2(2)) .* this.vFeatL2;
   
   % Adapt value range to achieve a correct contrast response function 
   inputScaled = (this.vE*input).^this.pE;

   % Normalization needs a fixed factor to achieve a firing rate of 1. 
   % Should be:(1+sigma/(maximal possible amplification*excitation))
   rMaxFactorL4 = (1+this.sigmaL4./(1+this.vSP+this.vFeatL4));            
   rMaxFactorL2 = (1+this.sigmaL2); 
   
   for t=1:this.tEnd
      
      %% Feedback from layer 2/3 to layer 4
      % - For speed reasons, we use the same intermediate variable for amplification and for suppression.
      % - Firstly, extend L2
      rL2Ext(:, this.rfH_FB+1:this.resL2(2)+this.rfH_FB) = rL2;
      % - Secondly, pool it
      for x=1:this.resL4(2)
         patch = rL2Ext(:,this.rfX_FB(1,x):this.rfX_FB(2,x));     
         rFB(:,x) = maxNonLin(this, patch .* this.wFB_2D{x} , [], 2);
      end            
      
      %% Feature-based suppression from layer 2/3 to layer 4
      supFeat(:) = 0;
      if(this.resL4(1) > 1)
         tmp = (rFB .* this.vSupFeat) .^this.pSupFeat;
         
         for i1 = 1:this.resL4(1)     % Loop over all features
            for i2 = 1:this.resL4(1)  % Loop over all adjacent features
               dist = abs(i1-i2) + 1; % Count from 1 to work with Matlab indexing
               supFeat(i1,:) = supFeat(i1,:) + this.wSupFeat(dist) * tmp(i2,:);
            end
         end         
      end
      
     
      %% Surround suppression from layer 2/3 to layer 4
      supSur(:) = 0;
      tmp = (rL2Ext * this.vSupSur).^this.pSupSur; 
      for xg=1:this.resL4(2)
         for xp = 1:this.resL2(2)+2*this.rfH_FB
            
            % Distance in L2 coordinates, whereby this.rfX_FB(3,xg) denotes rf center
            dist = abs(round(this.rfX_FB(3,xg) - xp));             
            if(dist > 0 && this.wSupSur(dist) ~= 0)
               supSur(:,xg) = supSur(:,xg) + tmp(:, xp) * this.wSupSur(dist);
            end
            
         end
      end
      
      %% Scaling of suppression
      % Goal: apply same relative amount of suppression independent of excitation
      % Idea:
      % - 1) Weight suppression with own excitation
      % - 2) Overall excitation also induces different rL2 activities which directly affects the amount of suppression.
      %      Thus estimate rL2 activity by own, un-modulated firing rate (r0) and also weight suppression 
      %      with it. In the paper, this is denoted as inverted divisive normalization function.
      % - 3) Assume that the suppression has an influence a and originates from a presynaptic neuron with activity r0: S = a * r0
      %      If so, the suppression would have the same influence if we multiply it with E./r0^2. This can be proofed by transfering 
      %      the firing rate equations.      
      r0 = rMaxFactorL4 .* inputScaled./(this.sigmaL4 + inputScaled);  % Unmodulated response      
      eps = 0.01;
      supCorrection = inputScaled .* ( (1+eps)./((r0 .* r0 + eps)) );  % Correction term: b = E./(r0^2)      

      %% Layer 4 calculations
      % Sum amplification 
      att = (1 + aSpatial_2D + this.vFeatL4*rFB); 
      
      % Firing rate        
      G = rMaxFactorL4.* inputScaled .*att./ (this.sigmaL4 + inputScaled .* att + supCorrection .* ( supSur + supFeat));
      rL4 = rL4 + this.hL4/this.tau * (G  - rL4);
        
      
      %% Excitation from layer 4 to layer 2/3
      % - Firstly, extend L4
      rL4Ext(:, this.rfH_FF+1:this.resL4(2)+this.rfH_FF) = rL4;      
      % - Secondly, pool it
      for x=1:this.resL2(2)
         patch = rL4Ext(:,this.rfX_FF(1,x):this.rfX_FF(2,x));         
         excL2(:,x) = maxNonLin(this, patch .* this.wFF_2D, [], 2);
      end          
      
      %% Layer 2/3 calculations
      %Sum amplification and apply it to excitation
      attPool = (1 + aFeat_2D);
      excAttL2 = excL2 .* attPool;
      
      % Firing rate
      G = rMaxFactorL2 .* excAttL2 ./ (this.sigmaL2 + excAttL2);      
      rL2 = rL2 + this.hL2/this.tau * ( G - rL2);      
      rL2(rL2>1) = 1;
      
   end
   
   % Add baseline which is ignored for amplification and suppression 
   rL4 = this.baseline + (1-this.baseline)*rL4;
   rL2 = this.baseline + (1-this.baseline)*rL2;
   
   % Calculate suppression variables for return
   supFeatRet = supCorrection .* supFeat;
   supSpatialRet = supCorrection .*  supSur;
   
   %% Plotting of neuronal responses
   if(plotResponses)
      figure
      ysub = 3;
      xsub = 3;
      cscale = [0,1];
      
      subplot(ysub, xsub, 1);
      imagesc(aFeat_2D, cscale);
      title('A^{FEAT-L2}');
      
      subplot(ysub, xsub, 4);
      imagesc(aSpatial_2D, cscale);
      title('A^{SP}');
      
      subplot(ysub, xsub, [2,3]);
      imagesc(rL2, cscale);
      title('Layer 2/3');
      
      subplot(ysub, xsub, [5,6]);
      imagesc(rL4, cscale);
      title('Layer 4');
      
      subplot(ysub, xsub, [8,9]);
      imagesc(input, cscale);
      title('Input');
      ylabel('Features');
      xlabel('Locations');      
   end
   
end

