% Demonstrate hyperdimensional encoding implementation of the SeqSLAM core
%
% VSA tutorial ECAI 2020
%
% Peer Neubert 
% peer.neubert@etit.tu-chemnitz.de
%
function demo_hvSeqSLAM    

    % load data
    W = load('winter.mat','D'); % each row is a CNN descriptor
    S = load('summer.mat','D');    
    
    % simple pairwise comparison
    DD = pdist2(S.D, W.D, 'cosine');
    evaluate(DD, 0);
    
    % initilize random number generator for repeatability
    rng(0);
        
    % run different sequence length parameters
    for d = [1,2,3,4,5]
        
        % create VSA instance
        vsa = VSA_MAPC(size(W.D,2));
        
        % create random position hypervectors
        % the same positions have to be used for all encodings!
        pos = vsa.generate(2*d+1);
        
        % encode (this is the VSA part)
        SY = createSeqBundle(vsa, pos, S.D, d);
        WY = createSeqBundle(vsa, pos, W.D, d);     
        
        % evaluate
        DD = pdist2(SY, WY,'cosine');
    
        % evaluate
        evaluate(DD,d);
    end
end

% Hyperdimensional encoding of image descriptor sequences
%
% vsa ... vsa class instance with .bundle() and .bind() operators
% pos ... [2*ds+1,m] matrix, each row is position vector
% X ... [n,m] matrix, each row is place descriptor of length m
% d ... sequence length parameter, sequence length is 2*ds+1
%
% Y ... [n,m] matrix, each row is HD vecor encoding a sequence of lengh (2*ds+1)
function Y = createSeqBundle(vsa,pos,X,d)
    
    % initialize the new descriptor
    Y = zeros(size(X));
    
    % for each image accumulate sequence information in a new descriptor Yi
    for i=1:size(X,1)        
        
        % for each position in the sequence
        for k=-d:d            
            if i+k>0 && i+k<=size(X,1)
                
                curHV = X(i+k,:); % current descriptor vector
                curPos = pos(k+d+1,:); % current position vector

                % update the new descriptor: Yi = Yi + ( curHV*curPos )
                Y(i,:) = vsa.bundle( [Y(i,:); vsa.bind(curHV, curPos)]);
                
            end
        end
        
    end
end

function evaluate(DD, d)
    persistent h
    persistent names    
    
    % prepare visualization
    if isempty(h)        
        names = {};
        figure();
    end
    [P,R] = createPR(DD, eye(288)); 
    hold all;
    h(end+1) = plot(R,P);
    
    if d==0
        names{end+1} = sprintf('pairwise: d=%d', d);
    else
        names{end+1} = sprintf('HV SeqSLAM: d=%d', d);
    end
        
    legend(h,names, 'location','southwest');
    xlabel('Recall');
    ylabel('Precision');
    drawnow
    
end




















    
