% Encode a 2d pose in a {-1,1} vector using the binding of an x and y
% vector obtained from concatenating start and end base vectors for each
% dimension.
%
% P ... pose matrix, each row is [y,x], each in [0,1]
% 
% encodedP ... each row is {-1,1} encoded pose
% 
function encodedP = encodePosesHDCconcatMultiAttractor(P, params)

    %% prepare basis vectors
    nBaseX = params.nX+1;
    persistent X
    if isempty(X) || size(X,1) ~= nBaseX || size(X,2) ~= params.nDims
        rng(sum(uint8('poseX')));
        X = 1-2*(rand(nBaseX, params.nDims)>0.5);
    end
    posX = 0 : 1/(nBaseX-1) : 1;    
        
    nBaseY = params.nY+1;
    persistent Y
    if isempty(Y) || size(Y,1) ~= nBaseY || size(Y,2) ~= params.nDims
        rng(sum(uint8('poseY')));
        Y = 1-2*(rand(nBaseY, params.nDims)>0.5);   
    end
    posY = 0 : 1/(nBaseY-1) : 1;
    
    %% check relative poses
    xr = P(:,2);
    yr = P(:,1);        
    assert(all(xr>=0 & xr<=1));
    assert(all(yr>=0 & yr<=1));
        
    %% encode
    encodedP = zeros(size(P,1), params.nDims);
    for i=1:size(P,1)
    
        % find attractors and split index
        [Yidx1, Yidx2, YsplitIdx] =  findAttractorsAndSplitIdx(yr(i), posY, params.nDims);
        [Xidx1, Xidx2, XsplitIdx] =  findAttractorsAndSplitIdx(xr(i), posX, params.nDims);
        
        % apply
        xVec = [X(Xidx1, 1:XsplitIdx) X(Xidx2, XsplitIdx+1:end)];
        yVec = [Y(Yidx1, 1:YsplitIdx) Y(Yidx2, YsplitIdx+1:end)];
        
        % combine
        encodedP(i,:) = xVec .* yVec;
    end
           
end

% Find the indices of the neighboring base vectors and the split index for
% the concatenation
%
% p ... input pos, in [0,1]
% attractorP ... vector with a postion value 0...1 for each attractor
% nDIms ... number of dimensions
%
% idx1, idx2 ... indices of base vectors
% splitIdx ... index, where to split the base vectors
%
function [idx1, idx2, splitIdx] =  findAttractorsAndSplitIdx(p, attractorP, nDims)

    assert(p>=0);
    assert(p<=1);
    assert(~any(attractorP(:)<0));
    assert(~any(attractorP(:)>1));

    % find x attractor below (the largest pos that is smaller than p)
    idx1 = max(find(attractorP<=p));
        
    % find x attractor above (the smallest pos that is larger than p)
    idx2 = min(find(attractorP>=p));
        
    % find weighting
    d1 = abs(attractorP(idx1)-p);
    d2 = abs(attractorP(idx2)-p);
    w = d2/(d1+d2);    
    
    if isnan(w)
        w = 0;
    end
    
    % compute index from weighting 
    splitIdx = round(w*nDims);
    
end
