% Generate a global descriptor for each image based on a set of local
% features with positions
%
% Y... vector of local landmarks with the following properties:
%      Y(i).keypoints ... single matrix, each row (y,x)
%      Y(i).descriptors ... single matrix, each row is a descriptor
%      Y(i).imsize  ... image size [width height]
% params.nDims ... the number of vector dimensions, e.g. 4,096
% params.nX ... the number of grid cells in x-direction (this one less
%               than the number of base-vectors), e.g. 4
% params.nY ... the number of grid cells in y-direction (this one less
%               than the number of base-vectors), e.g. 6 
% params.nFeat ... the number of used keypoints, e.g. 200
%
% hdcD ... output descriptor matrix, each row is a descriptor
%
function hdcD = encodeImagesHDC(Y, params)
    
    nIm = numel(Y);
    hdcD = zeros(nIm, params.nDims);
    for i=1:nIm
        
        % we use params.nFeat local feature descriptors D
        nFeat = min(params.nFeat, size(Y(i).descriptors,1));
        D = Y(i).descriptors(1:nFeat,:);
        
        % we need poses P to be in range [0,1]
        P = Y(i).keypoints(1:nFeat,:) ./ single([Y(i).imsize(2), Y(i).imsize(1)]);
        P(isnan(P))=0;
        assert( all(P(:)>=0) & all(P(:)>=0));
        
        hdcD(i,:) = bundleLocalDescriptorsWithPoses(D, P, params);
    end
end

% Bind each pre-processed descriptor to its pose encoding and bundle
%
% D ... descriptor matrix, each row is descriptor
% P ... pose matrix, each row is pose [y,x]
%
% hdcD ... output descriptor, row vector of length params.nDims
%
function hdcD = bundleLocalDescriptorsWithPoses(D, P, params)
  
    %% descriptor pre-processing    
    % projection
    persistent A % this is the projection matrix
    if isempty(A) || size(A,2) ~= params.nDims
        rng(sum(uint8('projection')));
        A = randn(size(D,2), params.nDims);
        A = orth(A')'; % this is optional for large nDims
    end
    projD = D*A;

    % standardize descriptors per image        
    mu = mean(projD);
    sigma = std(projD);
    stdProjD = (projD - mu) ./ sigma;
    stdProjD(isnan(stdProjD)) = 0;
    
    %% encode poses
    encodedP = encodePosesHDCconcatMultiAttractor(P, params);
    
    %% bind each descriptor to its pose and bundle    
    hdcD = sum(stdProjD .* encodedP, 1);
end
