% Implementation of Alg. 1 "Inconsistency Resolution Procedure (IRP)" 
% from the paper. However, the implementation uses distances instead
% of similarities. The cosine similarities S from the paper can be easily
% converted to a distance D by D=1-S
% 
% Given a distance matrix D_AB of inter-set distances and a D_AA of intra
% set distances, this function creates a new distance matrix M without
% inconsistencies with respect to D_AA.
%
% D_AB ... pairwise distance matrix between sets A and B, a row in
%          D_AB corresponds to an element from A, a column to an element from B
% D_AA ... pariwise distance matrix within the set A
% preempt_rate ... maximum fraction of all elements of A that can be
%                  matched to an element of B (default is 1)
%
% M(i,j) ... updated distance matrix D_AB without inconsitencies with
%            regarding D_AA
%
%   =====================================================================
%   Copyright (C) 2021  Peer Neubert, peer.neubert@etit.tu-chemnitz.de
%   
%   This program is free software: you can redistribute it and/or modify
%   it under the terms of the GNU General Public License as published by
%   the Free Software Foundation, either version 3 of the License, or
%   (at your option) any later version.
%   
%   This program is distributed in the hope that it will be useful,
%   but WITHOUT ANY WARRANTY; without even the implied warranty of
%   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%   GNU General Public License for more details.
%   
%   You should have received a copy of the GNU General Public License
%   along with this program.  If not, see <http://www.gnu.org/licenses/>.
%   =====================================================================
% 
function M = inconsistencyResolutionProcedure(D_AB, D_AA, preempt_rate)
    
    if ~exist('preempt_rate', 'var'), preempt_rate = 1;end
    
    M = D_AB; 
        
    % for each element from B, check the incompatibilities
    nA=size(D_AB,1);
    for j=1:size(D_AB,2)

        % sort elements in A in ascending distance
        [ASortVals, ASortIdx] = sort(D_AB(:,j),'ascend');
                
        % this is the maximum distance in the current clique
        cliqueMax = -inf;
        
        % process potential matchings in sorted order
        for i=1:preempt_rate*nA
            % update the maximum distance in the clique extended with the
            % current matching
            newMax = max(D_AA(ASortIdx(1:i), ASortIdx(i)));
            cliqueMax = max( cliqueMax, newMax);            
            
            % update the distance of the current matching
            M(ASortIdx(i),j) = max( cliqueMax, ASortVals(i));
        end        
        
        % It is important to increase the distance for all other
        % comparisons (which are very likely false matchings) to avoid
        % false positives
        for i=i:nA
            M(ASortIdx(i),j) = max( cliqueMax, ASortVals(i));
        end
        
        
    end    
    
end